//
//  YHGCApplicationTypeSelectCell.swift
//  galaxy
//
//  Created by alexzzw on 2024/11/11.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

class YHGCApplicationTypeSelectCell: UITableViewCell {
    static let cellReuseIdentifier = "YHGCApplicationTypeSelectCell"
    private let selectedBorderColor: UIColor = .brandMainColor
    
    var actionBtnEvent: (() -> Void)?
    var isCurrentSelected: Bool = false {
        didSet {
            guard isCurrentSelected != oldValue else {
                return
            }
            selectIconView.image = isCurrentSelected ? UIImage(named: "gc_application_type_selected"): UIImage(named: "gc_application_type_unselected")
            subContainerView.lineWidth = isCurrentSelected ? 2 : nil
            subContainerView.lineColor = isCurrentSelected ? selectedBorderColor.cgColor : nil
        }
    }
    
    private lazy var subContainerView: YHBaseCornerRadiusBorderView = {
        let view = YHBaseCornerRadiusBorderView(cornerRadius: 6, corner: .allCorners, lineWidth: nil, lineColor: nil)
        view.backgroundColor = .white
        return view
    }()
    
    private lazy var infoTitleLabel: UILabel = {
        let label = UILabel()
        label.font = .PFSC_B(ofSize: 17)
        label.textColor = .mainTextColor
        return label
    }()
    
    private lazy var infoDetailLabel: UILabel = {
        let label = UILabel()
        label.textColor = UIColor(hexString: "#6D788A")
        label.font = .PFSC_R(ofSize: 13)
        label.lineBreakMode = .byCharWrapping
        label.numberOfLines = 3
        return label
    }()
    
    private lazy var iconView: UIImageView = {
        let imageView = UIImageView()
        return imageView
    }()
    
    private lazy var selectIconView: UIImageView = {
        let imageView = UIImageView()
        imageView.image = UIImage(named: "gc_application_type_unselected")
        return imageView
    }()
    
    private lazy var actionBtn: UIButton = {
        let button = UIButton(type: .custom)
        button.setTitle("申请标准", for: .normal)
        button.setTitleColor(.mainTextColor, for: .normal)
        button.titleLabel?.font = .PFSC_R(ofSize: 14)
        button.setImage(UIImage(named: "right_arrow_black_20"), for: .normal)
        button.addTarget(self, action: #selector(actionBtnClicked), for: .touchUpInside)
        return button
    }()
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        setupUI()
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    func setupCellInfo(type: YHGCApplicationType, isCurrentSelected: Bool) {
        infoTitleLabel.text = type.titleString
        infoDetailLabel.text = type.detailString
        iconView.image = UIImage(named: type.iconString())
        self.isCurrentSelected = isCurrentSelected
    }
}

extension YHGCApplicationTypeSelectCell {
    @objc private func actionBtnClicked() {
        actionBtnEvent?()
    }
    
    private func setupUI() {
        selectionStyle = .none
        backgroundColor = .clear
        contentView.addSubview(subContainerView)
        subContainerView.snp.makeConstraints { make in
            make.top.bottom.equalToSuperview()
            make.left.equalToSuperview().offset(20)
            make.right.equalToSuperview().offset(-20)
        }
        subContainerView.addSubview(iconView)
        subContainerView.addSubview(infoTitleLabel)
        subContainerView.addSubview(selectIconView)
        subContainerView.addSubview(infoDetailLabel)
        subContainerView.addSubview(actionBtn)
        
        iconView.setContentCompressionResistancePriority(.required, for: .horizontal)
        infoTitleLabel.setContentCompressionResistancePriority(.defaultLow, for: .horizontal)
        selectIconView.setContentCompressionResistancePriority(.defaultHigh, for: .horizontal)
        infoDetailLabel.setContentCompressionResistancePriority(.defaultLow, for: .horizontal)
        
        actionBtn.setContentCompressionResistancePriority(.required, for: .vertical)
        infoDetailLabel.setContentCompressionResistancePriority(.defaultLow, for: .vertical)
        
        iconView.snp.makeConstraints { make in
            make.top.equalToSuperview().offset(20)
            make.left.equalToSuperview().offset(20)
            make.width.equalTo(88)
            make.height.equalTo(108).priority(.high)
            make.bottom.lessThanOrEqualToSuperview().offset(-20)
        }
        
        infoTitleLabel.snp.makeConstraints { make in
            make.left.equalTo(iconView.snp.right).offset(20)
            make.top.equalTo(iconView.snp.top)
        }
        
        selectIconView.snp.makeConstraints { make in
            make.right.equalToSuperview().offset(-20)
            make.top.equalTo(iconView.snp.top)
            make.left.greaterThanOrEqualTo(infoTitleLabel.snp.right).offset(20)
        }
        
        infoDetailLabel.snp.makeConstraints { make in
            make.left.equalTo(infoTitleLabel.snp.left)
            make.top.equalTo(infoTitleLabel.snp.bottom).offset(8)
            make.right.lessThanOrEqualToSuperview().offset(-20)
        }
        
        actionBtn.snp.makeConstraints { make in
            make.left.equalTo(infoTitleLabel.snp.left)
            make.top.greaterThanOrEqualTo(infoDetailLabel.snp.bottom).offset(0).priority(.high)
            make.bottom.equalTo(iconView.snp.bottom)
        }
        
        actionBtn.iconInRight(with: 0)
        
    }
}

private extension YHGCApplicationType {
    func iconString() -> String {
        switch self {
        case .typeA:
            return "gc_application_type_a"
        case .typeB:
            return "gc_application_type_b"
        case .typeC:
            return "gc_application_type_c"
        }
    }
}
