//
//  YHCirclePublishViewController.swift
//  galaxy
//
//  Created by alexzzw on 2025/9/24.
//  Copyright © 2025 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit
import SnapKit

class YHCirclePublishViewController: YHBaseViewController {
    private let marginX: CGFloat = 24
    private let itemSpace: CGFloat = 8
    
    var completion: (() -> Void)?
    
    // MARK: - Navigation Items
    private lazy var leftBarItem: UIBarButtonItem = {
        let item = UIBarButtonItem(image: UIImage(named: "nav_black_24")?.withRenderingMode(.alwaysOriginal), style: .plain, target: self, action: #selector(cancelButtonTapped))
        return item
    }()
    
    private lazy var rightButton: UIButton = {
        let button = UIButton(type: .custom)
        button.frame = CGRect(x: 0, y: 0, width: 56, height: 32)
        button.layer.cornerRadius = 16
        button.clipsToBounds = true
        button.setTitle("发布", for: .normal)
        button.titleLabel?.font = UIFont.PFSC_M(ofSize: 12)
        button.setTitleColor(UIColor.brandGrayColor0, for: .normal)
        button.setTitleColor(UIColor.brandGrayColor0, for: .disabled)
        button.isEnabled = false
        button.backgroundColor = UIColor.brandGrayColor4
        button.addTarget(self, action: #selector(publishButtonTapped), for: .touchUpInside)
        return button
    }()
    
    // MARK: - UI Components
    private lazy var scrollView: UIScrollView = {
        let scrollView = UIScrollView()
        scrollView.backgroundColor = .white
        scrollView.showsVerticalScrollIndicator = false
        return scrollView
    }()
    
    private lazy var contentView: UIView = {
        let view = UIView()
        view.backgroundColor = .white
        return view
    }()
    
    private lazy var userInfoView: UIView = {
        let view = UIView()
        view.backgroundColor = .white
        return view
    }()
    
    private lazy var avatarImageView: UIImageView = {
        let imageView = UIImageView()
        imageView.image = UIImage(named: "people_head_default") // 设置默认头像
        imageView.layer.cornerRadius = 17
        imageView.clipsToBounds = true
        imageView.contentMode = .scaleAspectFill
        return imageView
    }()
    
    private lazy var usernameLabel: UILabel = {
        let label = UILabel()
        label.text = "Monica杨晓丽"
        label.font = UIFont.PFSC_R(ofSize: 14)
        label.textColor = .brandGrayColor8
        return label
    }()
    
    private lazy var subtitleLabel: UILabel = {
        let label = UILabel()
        label.text = "董事长"
        label.font = UIFont.PFSC_R(ofSize: 11)
        label.textColor = UIColor.brandGrayColor6
        return label
    }()
    
    private lazy var textView: UITextView = {
        let textView = UITextView()
        textView.font = UIFont.PFSC_B(ofSize: 17)
        textView.textColor = .brandGrayColor8
        textView.backgroundColor = .clear
        textView.delegate = self
        textView.textContainer.lineFragmentPadding = 0
        textView.textContainerInset = UIEdgeInsets.zero
        textView.showsVerticalScrollIndicator = false
        return textView
    }()
    
    private lazy var placeholderLabel: UILabel = {
        let label = UILabel()
        label.text = "添加标题"
        label.font = UIFont.PFSC_R(ofSize: 17)
        label.textColor = UIColor.brandGrayColor5
        return label
    }()
    
    private lazy var detailTextView: UITextView = {
        let textView = UITextView()
        textView.font = UIFont.PFSC_R(ofSize: 14)
        textView.textColor = UIColor.brandGrayColor8
        textView.backgroundColor = .clear
        textView.delegate = self
        textView.textContainer.lineFragmentPadding = 0
        textView.textContainerInset = UIEdgeInsets.zero
        textView.showsVerticalScrollIndicator = false
        return textView
    }()
    
    private lazy var detailPlaceholderLabel: UILabel = {
        let label = UILabel()
        label.text = "分享生活，表达想法，随时随地..."
        label.font = UIFont.PFSC_R(ofSize: 14)
        label.textColor = UIColor.brandGrayColor5
        label.numberOfLines = 0
        return label
    }()
    
    private lazy var mediaCollectionView: UICollectionView = {
        let layout = UICollectionViewFlowLayout()
        layout.minimumInteritemSpacing = itemSpace
        layout.minimumLineSpacing = itemSpace
        let itemWidth = getItemWidth()
        layout.itemSize = CGSize(width: itemWidth, height: itemWidth)
        
        let collectionView = UICollectionView(frame: .zero, collectionViewLayout: layout)
        collectionView.backgroundColor = .white
        collectionView.delegate = self
        collectionView.dataSource = self
        collectionView.register(YHCircleMediaCell.self, forCellWithReuseIdentifier: "YHCircleMediaCell")
        collectionView.register(YHCircleAddPhotoCell.self, forCellWithReuseIdentifier: "YHCircleAddPhotoCell")
        collectionView.showsVerticalScrollIndicator = false
        return collectionView
    }()
    
    private lazy var divideLine: UIView = {
        let view = UIView()
        view.backgroundColor = UIColor.brandGrayColor3
        return view
    }()
    
    // MARK: - Properties
    private var mediaItems: [YHSelectMediaItem] = []
    private let maxMediaCount = 9
    
    override func viewDidLoad() {
        super.viewDidLoad()
        setupNav()
        setupUI()
        setupConstraints()
        setupNotifications()
    }
    
    deinit {
        NotificationCenter.default.removeObserver(self)
        printLog("YHCirclePublishViewController deinit")
    }
    
    // MARK: - Setup Methods
    private func setupNav() {
        gk_navTitle = "发布动态"
        gk_navLeftBarButtonItem = leftBarItem
        gk_navRightBarButtonItem = UIBarButtonItem(customView: rightButton)
        gk_navItemRightSpace = 16
        gk_navItemLeftSpace = 16
        gk_navBackgroundColor = .white
    }
    
    private func setupUI() {
        view.backgroundColor = .white
        
        view.addSubview(scrollView)
        scrollView.addSubview(contentView)
        
        contentView.addSubview(userInfoView)
        userInfoView.addSubview(avatarImageView)
        userInfoView.addSubview(usernameLabel)
        userInfoView.addSubview(subtitleLabel)
        
        contentView.addSubview(textView)
        contentView.addSubview(placeholderLabel)
        contentView.addSubview(divideLine)
        contentView.addSubview(detailTextView)
        contentView.addSubview(detailPlaceholderLabel)
        contentView.addSubview(mediaCollectionView)
    }
    
    private func setupConstraints() {
        scrollView.snp.makeConstraints { make in
            make.top.equalTo(k_Height_NavigationtBarAndStatuBar)
            make.left.right.bottom.equalToSuperview()
        }
        
        contentView.snp.makeConstraints { make in
            make.edges.equalToSuperview()
            make.width.equalTo(view.snp.width)
        }
        
        userInfoView.snp.makeConstraints { make in
            make.top.equalToSuperview().offset(20)
            make.left.right.equalToSuperview().inset(marginX)
        }
        
        avatarImageView.snp.makeConstraints { make in
            make.left.top.equalToSuperview()
            make.width.height.equalTo(34)
            make.bottom.equalToSuperview()
        }
        
        usernameLabel.snp.makeConstraints { make in
            make.left.equalTo(avatarImageView.snp.right).offset(8)
            make.top.equalTo(avatarImageView.snp.top)
            make.right.equalToSuperview()
        }
        
        subtitleLabel.snp.makeConstraints { make in
            make.left.equalTo(usernameLabel)
            make.bottom.equalTo(avatarImageView.snp.bottom)
            make.right.equalToSuperview()
        }
        
        textView.snp.makeConstraints { make in
            make.top.equalTo(userInfoView.snp.bottom).offset(24)
            make.left.right.equalToSuperview().inset(marginX)
            make.height.greaterThanOrEqualTo(24)
        }
        
        placeholderLabel.snp.makeConstraints { make in
            make.top.left.equalTo(textView)
        }
        
        divideLine.snp.makeConstraints { make in
            make.top.equalTo(textView.snp.bottom).offset(16)
            make.left.right.equalTo(textView)
            make.height.equalTo(1)
        }
        
        detailTextView.snp.makeConstraints { make in
            make.top.equalTo(divideLine.snp.bottom).offset(16)
            make.left.right.equalToSuperview().inset(marginX)
            make.height.greaterThanOrEqualTo(20)
        }
        
        detailPlaceholderLabel.snp.makeConstraints { make in
            make.top.left.equalTo(detailTextView)
        }
        
        mediaCollectionView.snp.makeConstraints { make in
            make.top.equalTo(detailTextView.snp.bottom).offset(96)
            make.left.right.equalToSuperview().inset(marginX)
            make.height.equalTo(200)
            make.bottom.equalToSuperview().offset(-30)
        }
        
        updateCollectionViewHeight()
    }
    
    private func setupNotifications() {
        NotificationCenter.default.addObserver(self, selector: #selector(keyboardWillShow), name: UIResponder.keyboardWillShowNotification, object: nil)
        NotificationCenter.default.addObserver(self, selector: #selector(keyboardWillHide), name: UIResponder.keyboardWillHideNotification, object: nil)
    }
    
    private func getItemWidth() -> CGFloat {
        let itemWidth = (UIScreen.main.bounds.width - marginX * 2 - itemSpace * 2) / 3
        return itemWidth
    }
    
    // MARK: - Actions
    @objc private func cancelButtonTapped() {
        if hasContent() {
            showCancelAlert()
        } else {
            dismiss(animated: true)
        }
    }
    
    @objc private func publishButtonTapped() {
        // 发布逻辑
        showPublishingAlert()
        
        // 模拟发布延迟
        DispatchQueue.main.asyncAfter(deadline: .now() + 2.0) {
            self.completion?()
            self.dismiss(animated: true)
        }
    }
    
    // MARK: - Keyboard Handling
    @objc private func keyboardWillShow(notification: NSNotification) {
        guard let keyboardFrame = notification.userInfo?[UIResponder.keyboardFrameEndUserInfoKey] as? CGRect,
              let duration = notification.userInfo?[UIResponder.keyboardAnimationDurationUserInfoKey] as? Double else { return }
        
        let keyboardHeight = keyboardFrame.height
        
        UIView.animate(withDuration: duration) {
            self.scrollView.contentInset.bottom = keyboardHeight
            self.scrollView.verticalScrollIndicatorInsets.bottom = keyboardHeight
        }
    }
    
    @objc private func keyboardWillHide(notification: NSNotification) {
        guard let duration = notification.userInfo?[UIResponder.keyboardAnimationDurationUserInfoKey] as? Double else { return }
        
        UIView.animate(withDuration: duration) {
            self.scrollView.contentInset.bottom = 0
            self.scrollView.verticalScrollIndicatorInsets.bottom = 0
        }
    }
    
    // MARK: - Helper Methods
    private func hasContent() -> Bool {
        return !textView.text.isEmpty || !detailTextView.text.isEmpty || !mediaItems.isEmpty
    }
    
    private func updatePublishButton() {
        let hasContent = hasContent()
        rightButton.isEnabled = hasContent
        rightButton.backgroundColor = hasContent ? UIColor.brandGrayColor8 : UIColor.brandGrayColor4
    }
    
    private func updateCollectionViewHeight() {
        let rows = ceil(Double(mediaItems.count + 1) / 3.0)
        let itemWidth = getItemWidth()
        let height = CGFloat(rows) * itemWidth + CGFloat(max(0, rows - 1)) * 8
        
        mediaCollectionView.snp.updateConstraints { make in
            make.height.equalTo(max(height, itemWidth))
        }
    }
    
    private func showCancelAlert() {
        let alert = UIAlertController(title: "确定要离开吗？", message: "离开后内容将不会保存", preferredStyle: .alert)
        alert.addAction(UIAlertAction(title: "取消", style: .cancel))
        alert.addAction(UIAlertAction(title: "确定", style: .destructive) { _ in
            self.dismiss(animated: true)
        })
        present(alert, animated: true)
    }
    
    private func showPublishingAlert() {
        let alert = UIAlertController(title: "发布中...", message: nil, preferredStyle: .alert)
        present(alert, animated: true)
        
        DispatchQueue.main.asyncAfter(deadline: .now() + 1.5) {
            alert.dismiss(animated: true)
        }
    }
}

// MARK: - UITextViewDelegate
extension YHCirclePublishViewController: UITextViewDelegate {
    func textViewDidChange(_ textView: UITextView) {
        if textView == self.textView {
            placeholderLabel.isHidden = !textView.text.isEmpty
        } else if textView == detailTextView {
            detailPlaceholderLabel.isHidden = !textView.text.isEmpty
        }
        updatePublishButton()
    }
    
    func textView(_ textView: UITextView, shouldChangeTextIn range: NSRange, replacementText text: String) -> Bool {
        if textView == self.textView {
            let currentText = textView.text ?? ""
            guard let stringRange = Range(range, in: currentText) else { return false }
            let updatedText = currentText.replacingCharacters(in: stringRange, with: text)
            return updatedText.count <= 100
        }
        return true
    }
}

// MARK: - UICollectionViewDataSource & Delegate
extension YHCirclePublishViewController: UICollectionViewDataSource, UICollectionViewDelegate {
    func collectionView(_ collectionView: UICollectionView, numberOfItemsInSection section: Int) -> Int {
        return min(mediaItems.count + 1, maxMediaCount)
    }
    
    func collectionView(_ collectionView: UICollectionView, cellForItemAt indexPath: IndexPath) -> UICollectionViewCell {
        if indexPath.item < mediaItems.count {
            if let cell = collectionView.dequeueReusableCell(withReuseIdentifier: "YHCircleMediaCell", for: indexPath) as? YHCircleMediaCell {
                cell.configure(with: mediaItems[indexPath.item])
                cell.deleteCallback = { [weak self] in
                    self?.removeMedia(at: indexPath.item)
                }
                return cell
            }
        } else {
            if let cell = collectionView.dequeueReusableCell(withReuseIdentifier: "YHCircleAddPhotoCell", for: indexPath) as? YHCircleAddPhotoCell {
                return cell
            }
        }
        return UICollectionViewCell()
    }
    
    func collectionView(_ collectionView: UICollectionView, didSelectItemAt indexPath: IndexPath) {
        if indexPath.item >= mediaItems.count {
            showMediaUploadSheet()
        } else {
            // 预览本地选中的媒体
            previewLocalMedia(at: indexPath.item)
        }
    }
    
    // 预览本地选择的媒体文件
    private func previewLocalMedia(at index: Int) {
        
        YHPictureReviewManager.shared.showLocalMedia(
            curIndex: index,
            selectMediaItems: self.mediaItems
        ) { [weak self] deletedIndex in
            // 删除回调处理
            guard let self = self else { return }
            
            DispatchQueue.main.async {
                // 从本地媒体数组中删除对应项
                if deletedIndex < self.mediaItems.count {
                    self.mediaItems.remove(at: deletedIndex)
                    
                    // 刷新集合视图
                    self.mediaCollectionView.reloadData()
                    self.updateCollectionViewHeight()
                    self.updatePublishButton()
                    
                    printLog("已从发布页面删除媒体项，剩余: \(self.mediaItems.count)")
                }
            }
        }
    }
    
    private func removeMedia(at index: Int) {
        mediaItems.remove(at: index)
        mediaCollectionView.reloadData()
        updateCollectionViewHeight()
        updatePublishButton()
    }
    
    private func showMediaUploadSheet() {
        YHMediaUploadSheetView.sheetView().show { [weak self] selectedMediaItems in
            guard let self = self else { return }
            
            // 检查是否超出最大数量限制
            let remainingSlots = maxMediaCount - mediaItems.count
            let itemsToAdd = Array(selectedMediaItems.prefix(remainingSlots))
            
            if itemsToAdd.count < selectedMediaItems.count {
                YHHUD.flash(message: "最多只能选择\(maxMediaCount)个媒体文件")
            }
            
            // 添加新选择的媒体项
            mediaItems.append(contentsOf: itemsToAdd)
            
            mediaCollectionView.reloadData()
            updateCollectionViewHeight()
            updatePublishButton()
            
            print("获得 \(itemsToAdd.count) 个媒体文件")
        }
    }
}
