//
//  YHResourceListModel.swift
//  galaxy
//
//  Created by alexzzw on 2025/9/24.
//  Copyright © 2025 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit
import SmartCodable

// MARK: - 资源列表模型
class YHResourceListModel: SmartCodable {
    
    required init() {}
    
    var needNewLine: Bool?
    
    // MARK: - 基本信息
    var id: String = ""
    var title: String = ""
    var content: String = ""
    var company_name: String = ""
    var company_logo: String = ""
    var contact_name: String = ""
    var contact_phone: String = ""
    var contact_wechat: String = ""
    var category_id: String = ""
    var category_name: String = ""
    var type: String = "" // service 或 demand
    var status: Int = 0 // 0-待审核 1-已发布 2-已下架
    var is_favorite: Bool = false
    var favorite_count: Int = 0
    var view_count: Int = 0
    var created_time: String = ""
    var updated_time: String = ""
    var expires_time: String = ""
    var images: [String] = []
    var tags: [String] = []
    var location: String = ""
    var price: String = ""
    var price_unit: String = ""
    
    // MARK: - 扩展属性
    var user_id: String = ""
    var user_name: String = ""
    var user_avatar: String = ""
    var certification_status: Int = 0 // 0-未认证 1-已认证
    var priority: Int = 0 // 优先级，用于排序
    var click_count: Int = 0 // 点击量
    var comment_count: Int = 0 // 评论数
    var share_count: Int = 0 // 分享数
    var service_area: String = "" // 服务区域
    var min_price: String = "" // 最低价格
    var max_price: String = "" // 最高价格
    var service_duration: String = "" // 服务周期
    var qualification: String = "" // 资质证明
    var business_hours: String = "" // 营业时间
    var website: String = "" // 公司网站
    var email: String = "" // 邮箱
    var address: String = "" // 详细地址
    
    // MARK: - UI相关属性
    var cell_width: CGFloat = 0
    var cell_height: CGFloat = 0
    
    // MARK: - 计算属性
    
    /// 显示时间（相对时间格式）
    var displayTime: String {
        return created_time.toDisplayTime()
    }
    
    /// 显示价格
    var displayPrice: String {
        if price.isEmpty {
            if !min_price.isEmpty && !max_price.isEmpty {
                return "\(min_price)-\(max_price)" + (price_unit.isEmpty ? "" : "/" + price_unit)
            } else if !min_price.isEmpty {
                return "￥\(min_price)起" + (price_unit.isEmpty ? "" : "/" + price_unit)
            }
            return "面议"
        }
        return "￥" + price + (price_unit.isEmpty ? "" : "/" + price_unit)
    }
    
    /// 类型显示名称
    var typeDisplayName: String {
        return type == "service" ? "企业服务" : "企业需求"
    }
    
    /// 状态显示名称
    var statusDisplayName: String {
        switch status {
        case 0:
            return "待审核"
        case 1:
            return "已发布"
        case 2:
            return "已下架"
        default:
            return "未知"
        }
    }
    
    /// 认证状态显示
    var certificationDisplayName: String {
        return certification_status == 1 ? "已认证" : "未认证"
    }
    
    /// 是否已认证
    var isCertified: Bool {
        return certification_status == 1
    }
    
    /// 是否为服务类型
    var isService: Bool {
        return type == "service"
    }
    
    /// 是否为需求类型
    var isDemand: Bool {
        return type == "demand"
    }
    
    /// 是否已过期
    var isExpired: Bool {
        guard !expires_time.isEmpty else { return false }
        let formatter = DateFormatter()
        formatter.dateFormat = "yyyy-MM-dd HH:mm:ss"
        guard let expireDate = formatter.date(from: expires_time) else { return false }
        return Date() > expireDate
    }
    
    /// 剩余天数
    var remainingDays: Int {
        guard !expires_time.isEmpty else { return -1 }
        let formatter = DateFormatter()
        formatter.dateFormat = "yyyy-MM-dd HH:mm:ss"
        guard let expireDate = formatter.date(from: expires_time) else { return -1 }
        let calendar = Calendar.current
        let components = calendar.dateComponents([.day], from: Date(), to: expireDate)
        return components.day ?? -1
    }
    
    /// 热度值（综合点击、收藏、分享等）
    var hotScore: Int {
        return view_count + favorite_count * 2 + share_count * 3 + comment_count * 1
    }
    
    /// 第一张图片URL
    var firstImageUrl: String {
        return images.first ?? ""
    }
    
    /// 是否有图片
    var hasImages: Bool {
        return !images.isEmpty
    }
    
    /// 标签字符串（用逗号分隔）
    var tagsString: String {
        return tags.joined(separator: ", ")
    }
    
    // MARK: - 计算Cell高度方法
    func didFinishMapping() {
        if needNewLine == nil {
            needNewLine = calculateNeedNewLine()
        }
        
    }
    
    func calculateNeedNewLine() -> Bool {
        let company = company_name.count > 0 ? company_name : "-"
        let categoryName = category_name.count > 0 ? category_name : "-"
        
        let companyWidth = YHResourceTableViewCell.getLabelWidth(company)
        let categoryNameWidth = YHResourceTableViewCell.getLabelWidth(categoryName)
        return companyWidth + categoryNameWidth >= KScreenWidth - YHResourceTableViewCell.logoWidth - YHResourceTableViewCell.logoToRight - YHResourceTableViewCell.arrowWidth - 2 * YHResourceTableViewCell.marginX - YHResourceTableViewCell.widthVLine - YHResourceTableViewCell.marginBetweenVLine * 2
    }
    
    // MARK: - 便利方法
    
    /// 更新收藏状态
    func updateFavoriteStatus(_ isFavorite: Bool) {
        is_favorite = isFavorite
        if isFavorite {
            favorite_count += 1
        } else {
            favorite_count = max(0, favorite_count - 1)
        }
    }
    
    /// 增加浏览量
    func incrementViewCount() {
        view_count += 1
    }
    
    /// 增加分享数
    func incrementShareCount() {
        share_count += 1
    }
    
    /// 验证数据完整性
    func isValid() -> Bool {
        return !id.isEmpty && 
               !title.isEmpty && 
               !content.isEmpty && 
               !company_name.isEmpty && 
               !contact_name.isEmpty && 
               !contact_phone.isEmpty &&
               !category_id.isEmpty
    }
    
    /// 获取联系方式字符串
    func getContactInfo() -> String {
        var contactInfo = contact_name
        if !contact_phone.isEmpty {
            contactInfo += " \(contact_phone)"
        }
        if !contact_wechat.isEmpty {
            contactInfo += " 微信:\(contact_wechat)"
        }
        return contactInfo
    }
}

// MARK: - 资源分类模型
class YHResourceCategory: SmartCodable {
    
    required init() {}
    
    var id: String = ""
    var name: String = ""
    var icon: String = ""
    var sort: Int = 0
    var parent_id: String = ""
    var children: [YHResourceCategory] = []
    var description_text: String = ""
    var is_hot: Bool = false
    var resource_count: Int = 0
    
    // 计算属性
    var hasChildren: Bool {
        return !children.isEmpty
    }
    
    var isTopLevel: Bool {
        return parent_id.isEmpty
    }
}

// MARK: - String Extension
extension String {
    func toDisplayTime() -> String {
        let formatter = DateFormatter()
        formatter.dateFormat = "yyyy-MM-dd HH:mm:ss"
        
        guard let date = formatter.date(from: self) else {
            return self
        }
        
        let now = Date()
        let interval = now.timeIntervalSince(date)
        
        if interval < 60 {
            return "刚刚"
        } else if interval < 3600 {
            return "\(Int(interval / 60))分钟前"
        } else if interval < 86400 {
            return "\(Int(interval / 3600))小时前"
        } else if interval < 2592000 {
            return "\(Int(interval / 86400))天前"
        } else {
            formatter.dateFormat = "MM-dd"
            return formatter.string(from: date)
        }
    }
}
