//
//  YHCircleCollectionViewCell.swift
//  galaxy
//
//  Created by alexzzw on 2025/9/24.
//  Copyright © 2025 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

// MARK: - Cell实现
class YHCircleCollectionViewCell: UICollectionViewCell {
    
    static let cellReuseIdentifier = "YHCircleCollectionViewCell"
    
    var likeBlock: ((YHCircleListModel) -> Void)?
    
    var listModel: YHCircleListModel? {
        didSet {
            updateUI()
        }
    }
    
    private lazy var containerView: UIView = {
        let view = UIView()
        view.backgroundColor = .white
        view.layer.cornerRadius = 8
        view.layer.shadowColor = UIColor.black.cgColor
        view.layer.shadowOffset = CGSize(width: 0, height: 2)
        view.layer.shadowRadius = 4
        view.layer.shadowOpacity = 0.1
        return view
    }()
    
    private lazy var avatarImageView: UIImageView = {
        let iv = UIImageView()
        iv.layer.cornerRadius = 15
        iv.clipsToBounds = true
        iv.contentMode = .scaleAspectFill
        iv.backgroundColor = UIColor(hex: 0xF5F5F5)
        return iv
    }()
    
    private lazy var nameLabel: UILabel = {
        let label = UILabel()
        label.font = UIFont.PFSC_M(ofSize: 13)
        label.textColor = UIColor.mainTextColor
        return label
    }()
    
    private lazy var timeLabel: UILabel = {
        let label = UILabel()
        label.font = UIFont.PFSC_R(ofSize: 11)
        label.textColor = UIColor.yhGreyColor
        return label
    }()
    
    private lazy var contentLabel: UILabel = {
        let label = UILabel()
        label.font = UIFont.PFSC_R(ofSize: 14)
        label.textColor = UIColor.mainTextColor
        label.numberOfLines = 0
        return label
    }()
    
    private lazy var imageView: UIImageView = {
        let iv = UIImageView()
        iv.contentMode = .scaleAspectFill
        iv.clipsToBounds = true
        iv.layer.cornerRadius = 6
        iv.backgroundColor = UIColor(hex: 0xF5F5F5)
        return iv
    }()
    
    private lazy var likeButton: UIButton = {
        let btn = UIButton(type: .custom)
        btn.setImage(UIImage(named: "home_like_normal"), for: .normal)
        btn.setImage(UIImage(named: "home_like_selected"), for: .selected)
        btn.addTarget(self, action: #selector(likeButtonClicked), for: .touchUpInside)
        return btn
    }()
    
    private lazy var likeCountLabel: UILabel = {
        let label = UILabel()
        label.font = UIFont.PFSC_R(ofSize: 11)
        label.textColor = UIColor.yhGreyColor
        return label
    }()
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        setupUI()
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    private func setupUI() {
        contentView.addSubview(containerView)
        
        containerView.addSubview(avatarImageView)
        containerView.addSubview(nameLabel)
        containerView.addSubview(timeLabel)
        containerView.addSubview(contentLabel)
        containerView.addSubview(imageView)
        containerView.addSubview(likeButton)
        containerView.addSubview(likeCountLabel)
        
        containerView.snp.makeConstraints { make in
            make.edges.equalToSuperview().inset(4)
        }
        
        avatarImageView.snp.makeConstraints { make in
            make.left.top.equalToSuperview().offset(10)
            make.width.height.equalTo(30)
        }
        
        nameLabel.snp.makeConstraints { make in
            make.left.equalTo(avatarImageView.snp.right).offset(8)
            make.top.equalTo(avatarImageView)
            make.right.equalToSuperview().offset(-10)
        }
        
        timeLabel.snp.makeConstraints { make in
            make.left.equalTo(nameLabel)
            make.bottom.equalTo(avatarImageView)
        }
        
        contentLabel.snp.makeConstraints { make in
            make.left.right.equalToSuperview().inset(10)
            make.top.equalTo(avatarImageView.snp.bottom).offset(8)
        }
        
        imageView.snp.makeConstraints { make in
            make.left.right.equalToSuperview().inset(10)
            make.top.equalTo(contentLabel.snp.bottom).offset(8)
            make.height.equalTo(100)
        }
        
        likeButton.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(10)
            make.bottom.equalToSuperview().offset(-10)
            make.width.height.equalTo(16)
        }
        
        likeCountLabel.snp.makeConstraints { make in
            make.left.equalTo(likeButton.snp.right).offset(4)
            make.centerY.equalTo(likeButton)
        }
    }
    
    private func updateUI() {
        guard let model = listModel else { return }
        
        nameLabel.text = model.user_name
        timeLabel.text = model.create_time
        contentLabel.text = model.content
        
        // 设置头像
        // avatarImageView.yh_setImage(with: model.user_avatar)
        
        // 设置图片
        if !model.images.isEmpty {
            imageView.isHidden = false
            // imageView.yh_setImage(with: model.images.first)
        } else {
            imageView.isHidden = true
        }
        
        // 设置点赞状态
        likeButton.isSelected = model.is_like
        likeCountLabel.text = model.like_count > 0 ? "\(model.like_count)" : ""
        likeCountLabel.textColor = model.is_like ? UIColor.brandMainColor : UIColor.yhGreyColor
    }
    
    @objc private func likeButtonClicked() {
        guard let model = listModel else { return }
        likeBlock?(model)
    }
}
