//
//  YHSearchInfoCell.swift
//  galaxy
//
//  Created by edy on 2024/4/8.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit
import Kingfisher

class YHSearchInfoCell: UITableViewCell {

    static let cellReuseIdentifier = "YHSearchInfoCell"
    static let textFont = UIFont.PFSC_R(ofSize: 15)
    
    var model: YHHomeListModel = YHHomeListModel()
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        setupUI()
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    lazy var headImgView: UIImageView = {
        let imagV = UIImageView()
        imagV.contentMode = .scaleAspectFill
        imagV.clipsToBounds = true
        imagV.image = UIImage(named: "home_cell0_head_icon")
        return imagV
    }()
    
    lazy var videoPlayImgView: UIImageView = {
        let imagV = UIImageView()
        imagV.image = UIImage(named: "home_play_btn")
        imagV.isHidden = true
        return imagV
    }()
    
    lazy var titleLabel: UILabel =  {
        var label = UILabel()
        label.font = Self.textFont
        label.textAlignment = .left
        label.textColor = UIColor.mainTextColor
        label.numberOfLines = 2
        return label
    }()
    
    func updateModel(_ model: YHHomeListModel) {
        self.model = model
        titleLabel.text = model.title
        videoPlayImgView.isHidden = (model.type != 1)
        let url = URL(string: model.img_url)
        headImgView.kf.setImage(with: url, placeholder: UIImage(named: "global_default_image"))
    }
    
    func setupUI() {
         
        contentView.backgroundColor = .white
        contentView.addSubview(headImgView)
        headImgView.addSubview(videoPlayImgView)
        contentView.addSubview(titleLabel)
        
        headImgView.snp.makeConstraints { make in
            make.width.height.equalTo(70.0)
            make.centerY.equalToSuperview()
            make.left.equalToSuperview().offset(20)
        }
        
        videoPlayImgView.snp.makeConstraints { make in
            make.width.height.equalTo(20.0)
            make.center.equalToSuperview()
        }
        
        titleLabel.snp.makeConstraints { make in
            make.centerY.equalToSuperview()
            make.left.equalTo(headImgView.snp.right).offset(20)
            make.right.equalToSuperview().offset(-20)
        }
    }
    
    func renderHighLightText(_ text: String) {
        
        var attributedString = NSMutableAttributedString(string: self.model.title, attributes: [
            .foregroundColor: UIColor.mainTextColor,
            .font: Self.textFont
        ])
        if isEmptyString(text) {
            self.titleLabel.attributedText = attributedString
            return
        }
        
        // 设置高亮样式
        let highlightAttributes: [NSAttributedString.Key: Any] = [
            .foregroundColor: UIColor.brandMainColor,
            .font: Self.textFont
        ]
        
        let ranges = findOccurrencesOfString(text, inString: self.model.title)
        for range in ranges {
            let changeRange = NSRange(range, in: self.model.title)
            attributedString.addAttributes(highlightAttributes, range: changeRange)
        }
        self.titleLabel.attributedText = attributedString
    }
    
    func findOccurrencesOfString(_ searchString: String, inString string: String) -> [Range<String.Index>] {
        var occurrences: [Range<String.Index>] = []
        var searchStartIndex = string.startIndex
        
        while let range = string.range(of: searchString, options: [.caseInsensitive], range: searchStartIndex..<string.endIndex) {
            occurrences.append(range)
            searchStartIndex = range.upperBound
        }
        return occurrences
    }
    
}
