//
//  YHCircleCollectionViewCell.swift
//  galaxy
//
//  Created by alexzzw on 2025/9/24.
//  Copyright © 2025 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit
import Kingfisher

class YHCircleCollectionViewCell: UICollectionViewCell {
    
    static let cellReuseIdentifier = "YHCircleCollectionViewCell"
    
    var listModel: YHDynamicItem? {
        didSet {
            updateUI()
        }
    }
    
    var onImageHeightChanged: (() -> Void)?
    
    // MARK: - UI Components
    
    lazy var titleImageView: UIImageView = {
        let imageV = UIImageView(image: UIImage(named: "plan_product_default"))
        imageV.contentMode = .scaleAspectFill
        imageV.isUserInteractionEnabled = true
        imageV.clipsToBounds = true
        return imageV
    }()
    
    private lazy var avatarImageView: UIImageView = {
        let iv = UIImageView()
        iv.layer.cornerRadius = 8
        iv.clipsToBounds = true
        iv.contentMode = .scaleAspectFill
        return iv
    }()
    
    private lazy var nameLabel: UILabel = {
        let label = UILabel()
        label.font = UIFont.PFSC_R(ofSize: 12)
        label.textColor = UIColor.brandGrayColor6
        return label
    }()
    
    lazy var contentLabel: UILabel = {
        let label = UILabel()
        label.text = ""
        label.textColor = UIColor.mainTextColor
        label.font = UIFont.PFSC_R(ofSize: 15)
        label.numberOfLines = 2
        label.lineBreakMode = .byTruncatingTail
        return label
    }()
    
    lazy var subHoldView: UIView = {
        let view = UIView()
        view.layer.backgroundColor = UIColor.white.cgColor
        view.layer.borderWidth = 0.6
        view.layer.borderColor = UIColor(red: 0.943, green: 0.952, blue: 0.969, alpha: 1).cgColor
        return view
    }()
    
    // MARK: - Initialization
    
    required init?(coder: NSCoder) {
        super.init(coder: coder)
    }
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        setupUI()
    }
    
    // MARK: - Setup UI
    
    private func setupUI() {
        contentView.clipsToBounds = true
        contentView.addSubview(titleImageView)
        titleImageView.snp.makeConstraints { make in
            make.left.right.top.equalToSuperview()
            make.height.equalTo(210)
        }
        
        contentView.addSubview(subHoldView)
        subHoldView.snp.makeConstraints { make in
            make.top.equalTo(titleImageView.snp.bottom)
            make.bottom.left.right.equalToSuperview()
        }
        
        // 内容文本
        subHoldView.addSubview(contentLabel)
        contentLabel.snp.makeConstraints { make in
            make.top.equalToSuperview().offset(12)
            make.left.equalToSuperview().offset(8)
            make.right.equalToSuperview().offset(-8)
        }
        
        // 头像
        subHoldView.addSubview(avatarImageView)
        avatarImageView.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(8)
            make.top.equalTo(contentLabel.snp.bottom).offset(12)
            make.width.height.equalTo(16)
        }
        
        // 昵称
        subHoldView.addSubview(nameLabel)
        nameLabel.snp.makeConstraints { make in
            make.left.equalTo(avatarImageView.snp.right).offset(4)
            make.centerY.equalTo(avatarImageView)
            make.right.equalToSuperview().offset(-8)
        }
        
    }
    
    // MARK: - Update UI
    
    private func updateUI() {
        guard let model = listModel else { return }
        
        // 顶部图片
        titleImageView.snp.remakeConstraints { make in
            make.left.right.top.equalToSuperview()
            make.height.equalTo(model.img_height_use)
        }
        
        if let firstResource = model.firstImageResource, !firstResource.url.isEmpty, let url = URL(string: firstResource.url) {
            titleImageView.kf.setImage(with: url, placeholder: UIImage(named: "plan_product_default")) { result in
                switch result {
                case let .success(value):
                    DispatchQueue.global().async {
                        var needRefreshHeight = false
                        let size = value.image.size
                        let scale = size.width > 0 ? Double(size.height / size.width) : 0.0
                        if scale > 0 {
                            let width = (KScreenWidth - 47) / 2
                            let height = scale * width
                            if height >= model.maxImageHeight, model.img_height_use >= model.maxImageHeight {
                                return
                            }
                            if height <= model.minImageHeight, model.img_height_use <= model.minImageHeight {
                                return
                            }
                            if abs(height - model.img_height_use) > 10 {
                                model.calHeightParam(height)
                                needRefreshHeight = true
                            }
                        }
                        
                        DispatchQueue.main.async {
                            if needRefreshHeight {
                                self.titleImageView.snp.remakeConstraints { make in
                                    make.left.right.top.equalToSuperview()
                                    make.height.equalTo(model.img_height_use)
                                }
                                self.onImageHeightChanged?()
                            }
                        }
                    }
                    
                case .failure:
                    printLog("kf setImage failed")
                }
            }
        } else {
            titleImageView.image = UIImage(named: "plan_product_default")
        }
        
        // 用户信息
        nameLabel.text = model.nickname.isEmpty ? "-" : model.nickname
        
        // 头像
        if !model.avatar.isEmpty, let url = URL(string: model.avatar) {
            avatarImageView.kf.setImage(with: url, placeholder: UIImage(named: "people_head_default"))
        } else {
            avatarImageView.image = UIImage(named: "people_head_default")
        }
        
        // 内容文本
        contentLabel.text = model.content
        contentLabel.snp.removeConstraints()
        contentLabel.snp.remakeConstraints { make in
            make.top.equalToSuperview().offset(12)
            make.left.equalToSuperview().offset(8)
            make.right.equalToSuperview().offset(-8)
            make.height.equalTo(model.textH)
        }
    }
}
