//
//  YHLiveMessageCell.swift
//  galaxy
//
//  Created by alexzzw on 2024/11/25.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import AttributedString
import UIKit

class YHLiveMessageCell: UITableViewCell {
    static let reuseIdentifier = "YHLiveMessageCell"

    // MARK: - UI Components

    private let containerView: YHBaseDynamicCornerRadiusView = {
        let view = YHBaseDynamicCornerRadiusView(cornerRadius: 8, corner: .allCorners)
        view.backgroundColor = UIColor.black.withAlphaComponent(0.3)
        view.layer.cornerRadius = 8
        return view
    }()

    private let contentLabel: UILabel = {
        let label = UILabel()
        label.numberOfLines = 0
        label.lineBreakMode = .byCharWrapping
        return label
    }()

    // MARK: - Initialization

    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        setupUI()
    }

    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }

    private func setupUI() {
        backgroundColor = .clear
        selectionStyle = .none
        contentView.addSubview(containerView)
        containerView.addSubview(contentLabel)
        setupConstraints()
    }

    private func setupConstraints() {
        containerView.snp.makeConstraints { make in
            make.left.equalToSuperview()
            make.right.lessThanOrEqualToSuperview()
            make.top.equalToSuperview().offset(2)
            make.bottom.equalToSuperview().offset(-2)
        }
        contentLabel.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(8)
            make.right.equalToSuperview().offset(-8)
            make.top.equalToSuperview().offset(4)
            make.bottom.equalToSuperview().offset(-4)
        }
    }

    func configureNormalMessage(_ nickname: String, _ content: String, highlightTxt: String = "", isAnchor: Bool = false) {
        if highlightTxt.isEmpty {
            let nickAtt = isAnchor ? ASAttributedString(string: nickname + "：", .foreground(UIColor(hexString: "#FFE3BB") ?? UIColor.white), .font(UIFont.PFSC_R(ofSize: 13))) : ASAttributedString(string: nickname + "：", .foreground(UIColor.white.withAlphaComponent(0.65)), .font(UIFont.PFSC_R(ofSize: 13)))
            let contentAtt = ASAttributedString(string: content, .foreground(UIColor.white), .font(UIFont.PFSC_R(ofSize: 13)))
            contentLabel.attributed.text = nickAtt + contentAtt
        } else {
            
            let nickAtt = isAnchor ? ASAttributedString(string: nickname + "：", .foreground(UIColor(hexString: "#FFE3BB") ?? UIColor.white), .font(UIFont.PFSC_R(ofSize: 13))) : ASAttributedString(string: nickname + "：", .foreground(UIColor.white.withAlphaComponent(0.65)), .font(UIFont.PFSC_R(ofSize: 13)))
            
            var contentAtt = ASAttributedString(string: content, .foreground(UIColor.white), .font(UIFont.PFSC_R(ofSize: 13)))
            if content.starts(with: highlightTxt, caseSensitive: false) {
                contentAtt = ""
                var content2 = content
                let part1 = ASAttributedString(string: highlightTxt, .foreground(UIColor.white.withAlphaComponent(0.65)), .font(UIFont.PFSC_R(ofSize: 13)))
                
                let part2Str = content2.slice(at: highlightTxt.count)
                
                let part2 = ASAttributedString(string: part2Str, .foreground(UIColor.white), .font(UIFont.PFSC_R(ofSize: 13)))
                contentLabel.attributed.text = nickAtt + part1 + part2
            } else {
                contentLabel.attributed.text = nickAtt + contentAtt
            }
        }
    }
    
    func configureTipsMessage(_ tips: String) {
        let tipsIcon = ASAttributedString.init(.image(UIImage(named: "live_msg_tips_icon") ?? UIImage(), .custom(.offset(CGPoint(x: 0, y: -4.0)), size: .init(width: 32, height: 18))))
        let contentAtt = (tipsIcon + ASAttributedString(string: " \(tips)", .foreground(UIColor.white), .font(UIFont.PFSC_R(ofSize: 13))))
        contentLabel.attributed.text = contentAtt
    }
    
    func configureUserEnterRoomMessage(_ usename: String) {
        let contentAtt = (ASAttributedString(string: "\(usename)", .foreground(UIColor.white.withAlphaComponent(0.65)), .font(UIFont.PFSC_R(ofSize: 13))))
        
        let tipsAtt = ASAttributedString(string: " 进入直播间", .foreground(UIColor.white), .font(UIFont.PFSC_R(ofSize: 13)))
        contentLabel.attributed.text = contentAtt + tipsAtt
    }
}
