//
//  YHRecommendedOnLiveView.swift
//  galaxy
//
//  Created by alexzzw on 2024/12/29.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import Lottie
import UIKit

class YHRecommendedOnLiveView: UIView {
    var buttonClickEvent: (() -> Void)?
    
    private lazy var flagImageView: LottieAnimationView = {
        let view = LottieAnimationView(name: "live")
        view.backgroundColor = UIColor.black.withAlphaComponent(0.6)
        view.loopMode = .loop
        return view
    }()

    private lazy var coverImageView: UIImageView = {
        let view = UIImageView()
        view.contentMode = .scaleAspectFit
        view.isUserInteractionEnabled = true
        return view
    }()

    private lazy var avatarImageView: UIImageView = {
        let view = UIImageView()
        view.isUserInteractionEnabled = true
        view.contentMode = .scaleAspectFit
        view.clipsToBounds = true
        view.layer.cornerRadius = 10
        view.layer.borderColor = UIColor.white.cgColor
        view.layer.borderWidth = 0.5
        return view
    }()

    private lazy var bottomLayerView: UIView = {
        let view = UIView()
        return view
    }()

    private lazy var anchorLabel: UILabel = {
        let label = UILabel()
        label.font = .PFSC_R(ofSize: 12)
        label.textColor = .white
        return label
    }()
    
    private lazy var gradientLayer: CAGradientLayer = {
        let layer = CAGradientLayer()
        layer.colors = [
            UIColor.black.withAlphaComponent(0).cgColor,
            UIColor.black.withAlphaComponent(0.9).cgColor
        ]
        layer.startPoint = CGPoint(x: 0.5, y: 0) // 上边中点
        layer.endPoint = CGPoint(x: 0.5, y: 1) // 下边中点
        return layer
    }()

    override init(frame: CGRect) {
        super.init(frame: frame)
        setupUI()
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    override func layoutSubviews() {
        super.layoutSubviews()
        gradientLayer.frame = bottomLayerView.bounds
    }
    
    @objc private func tapButton(gestureRecognizer: UITapGestureRecognizer) {
        buttonClickEvent?()
    }

    private func setupUI() {
        addSubview(coverImageView)
        addSubview(bottomLayerView)
        addSubview(avatarImageView)
        addSubview(anchorLabel)
        addSubview(flagImageView)
        sendSubviewToBack(coverImageView)
        insertSubview(bottomLayerView, aboveSubview: coverImageView)
        insertSubview(anchorLabel, aboveSubview: bottomLayerView)
        insertSubview(avatarImageView, aboveSubview: bottomLayerView)
        coverImageView.snp.makeConstraints { make in
            make.edges.equalToSuperview()
        }
        flagImageView.snp.makeConstraints { make in
            make.width.height.equalTo(16)
            make.top.equalToSuperview().offset(8)
            make.left.equalToSuperview().offset(8)
        }
        bottomLayerView.snp.makeConstraints { make in
            make.left.right.bottom.equalToSuperview()
            make.height.equalTo(60)
        }
        avatarImageView.snp.makeConstraints { make in
            make.width.height.equalTo(20)
            make.bottom.equalToSuperview().offset(-12)
            make.left.equalToSuperview().offset(8)
        }
        anchorLabel.snp.makeConstraints { make in
            make.left.equalTo(avatarImageView.snp.right).offset(4)
            make.centerY.equalTo(avatarImageView)
            make.right.lessThanOrEqualToSuperview().offset(-8)
        }
        gradientLayer.frame = bottomLayerView.bounds
        bottomLayerView.layer.mask = gradientLayer
        flagImageView.play()
        let tap = UITapGestureRecognizer(target: self, action: #selector(tapButton(gestureRecognizer:)))
        addGestureRecognizer(tap)
    }

    func updateViews(avatar: String, anchorName: String, image: String) {
        if let url = URL(string: image) {
            coverImageView.kf.setImage(with: url, placeholder: UIImage(named: "global_default_image"))
        } else {
            coverImageView.image = UIImage(named: "global_default_image")
        }
        if let url = URL(string: avatar) {
            avatarImageView.kf.setImage(with: url, placeholder: UIImage(named: "global_default_image"))
        } else {
            avatarImageView.image = UIImage(named: "global_default_image")
        }
        anchorLabel.text = anchorName
    }
}
