//
//  YHHKImmigrationRecordsVC.swift
//  galaxy
//
//  Created by alexzzw on 2024/9/14.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit
import JXSegmentedView

class YHHKImmigrationRecordsVC: YHBaseViewController {
    
    private lazy var controllerOne: YHHKRecordsSummaryVC = YHHKRecordsSummaryVC(orderId: self.orderId)
    private lazy var controllerTwo: YHHKRecordsContentVC = YHHKRecordsContentVC(orderId: self.orderId, userType: .main)
    private lazy var controllerThree: YHHKRecordsContentVC = YHHKRecordsContentVC(orderId: self.orderId, userType: .spouse)
    private lazy var controllerFour: YHHKRecordsContentVC = YHHKRecordsContentVC(orderId: self.orderId, userType: .children)
    
    private lazy var arrItemTitles = ["汇总", "主申(0)", "配偶(0)", "子女(0)"]
    
    private lazy var segmentedView: JXSegmentedView = {
        let view = JXSegmentedView(frame: CGRect(x: 0, y: k_Height_safeAreaInsetsTop(), width: KScreenWidth, height: 53))
        let indicator = JXSegmentedIndicatorLineView()
        indicator.indicatorWidth = 16
        indicator.indicatorHeight = 2
        indicator.indicatorColor = UIColor.brandMainColor
        indicator.indicatorCornerRadius = 0
        view.indicators = [indicator]
        view.dataSource = segmentedDataSource
        view.delegate = self
        return view
    }()
    
    private lazy var segmentedDataSource: JXSegmentedDotDataSource = {
        let dataSource = JXSegmentedDotDataSource()
        dataSource.isTitleColorGradientEnabled = true
        dataSource.titles = arrItemTitles
        dataSource.titleNormalFont = UIFont.PFSC_R(ofSize: 14)
        dataSource.titleNormalColor = .labelTextColor2
        dataSource.titleSelectedFont = UIFont.PFSC_M(ofSize: 16)
        dataSource.titleSelectedColor = UIColor(hexString: "#000000")!
        dataSource.isItemSpacingAverageEnabled = false
        dataSource.itemWidth = JXSegmentedViewAutomaticDimension
        dataSource.itemSpacing = 24
        dataSource.dotStates = [false, false, false, false]
        dataSource.dotSize = CGSize(width: 6, height: 6)
        return dataSource
    }()
    
    private lazy var listContainerView: JXSegmentedListContainerView = {
        return JXSegmentedListContainerView(dataSource: self, type: .scrollView)
    }()
    
    private lazy var arrItemVCs: [JXSegmentedListContainerViewListDelegate] = {
        return [controllerOne, controllerTwo, controllerThree, controllerFour]
    }()
    
    private lazy var addNewButton: UIButton = {
        let button = UIButton(type: .custom)
        button.titleLabel?.font = UIFont.PFSC_M(ofSize: 16)
        let buttonName = "新增在港记录"
        button.setTitle(buttonName, for: .normal)
        button.setTitle(buttonName, for: .highlighted)
        button.setTitleColor( UIColor(hex: 0xffffff), for: .normal)
        button.setTitleColor( UIColor(hex: 0xffffff), for: .highlighted)
        button.addTarget(self, action: #selector(clickAddNewButton), for: .touchUpInside)
        button.layer.cornerRadius = kCornerRadius3
        button.clipsToBounds = true
        button.backgroundColor = .brandMainColor
        return button
    }()
    
    private lazy var bottomView: UIView = {
        let view = UIView()
        view.backgroundColor = .white
        return view
    }()
    
    private lazy var viewModel = YHHKImmigrationRecordsViewModel()
    
    private let orderId: Int
    init(orderId: Int) {
        self.orderId = orderId
        super.init(nibName: nil, bundle: nil)
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    override func viewDidLoad() {
        super.viewDidLoad()
        setupUI()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        requestData()
    }
}

extension YHHKImmigrationRecordsVC {
    private func setupUI() {
        gk_navBarAlpha = 1
        gk_navTitle = "在港记录列表"
        
        view.backgroundColor = .white
        view.addSubview(segmentedView)
        view.addSubview(bottomView)
        bottomView.addSubview(addNewButton)
        view.addSubview(listContainerView)
        
        segmentedView.snp.makeConstraints { make in
            make.left.right.equalToSuperview()
            make.top.equalTo(k_Height_NavigationtBarAndStatuBar)
            make.height.equalTo(53)
        }
        
        bottomView.snp.makeConstraints { make in
            make.left.right.equalToSuperview()
            make.bottom.equalTo(view.safeAreaLayoutGuide.snp.bottom)
            make.height.equalTo(64)
        }
        
        addNewButton.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(kMargin)
            make.right.equalToSuperview().offset(-kMargin)
            make.centerY.equalToSuperview()
            make.height.equalTo(48)
        }
        
        listContainerView.snp.makeConstraints { make in
            make.left.right.equalToSuperview()
            make.top.equalTo(segmentedView.snp.bottom)
            make.bottom.equalTo(bottomView.snp.top)
        }
        segmentedView.listContainer = listContainerView
    }
    
    private func requestData() {
        viewModel.getStayInHKRecordStatistics(orderId: orderId) { [weak self] recordStatistics, error in
            if let recordStatistics = recordStatistics {
                self?.updateVC(recordStatistics: recordStatistics)
            } else {
                printLog("YHHKImmigrationRecordsVC:  请求失败")
                if let errorMsg = error?.errorMsg, errorMsg.count > 0 {
                    YHHUD.flash(message: errorMsg)
                }
            }
        }
    }
    
    private func updateVC(recordStatistics: YHHKImmigrationRecordsStatisticsModel) {
        arrItemTitles = ["汇总", "主申(\(recordStatistics.main.sum_times))", "配偶(\(recordStatistics.spouse.sum_times))", "子女(\(recordStatistics.children.sum_times))"]
        segmentedDataSource.titles = arrItemTitles
        segmentedView.reloadDataWithoutListContainer()
    }
    
    @objc private func clickAddNewButton() {
        let vc = YHHKRecordsEditContentVC(orderId: orderId, pageType: .add)
        navigationController?.pushViewController(vc)
    }
}

extension YHHKImmigrationRecordsVC: JXSegmentedViewDelegate {
    
    func segmentedView(_ segmentedView: JXSegmentedView, didSelectedItemAt index: Int) {
        // 先更新数据源的数据
//        segmentedDataSource.dotStates[index] = false
        // 再调用reloadItem(at: index)
//        segmentedView.reloadItem(at: index)
    }
}

extension YHHKImmigrationRecordsVC: JXSegmentedListContainerViewDataSource {
    
    func numberOfLists(in listContainerView: JXSegmentedListContainerView) -> Int {
        if let titleDataSource = segmentedView.dataSource as? JXSegmentedBaseDataSource {
            return titleDataSource.dataSource.count
        }
        return 0
    }
    
    func listContainerView(_ listContainerView: JXSegmentedListContainerView, initListAt index: Int) -> any JXSegmentedListContainerViewListDelegate {
        return arrItemVCs[index]
    }
    
    //
}
