//
//  YHCircleHeaderButtonsView.swift
//  galaxy
//
//  Created by alexzzw on 2025/9/24.
//  Copyright © 2025 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

class YHHeaderButtonItemView: UIControl {
    
    private lazy var iconView: UIImageView = {
        let imageView = UIImageView()
        return imageView
    }()
    
    private lazy var contentLabel: UILabel = {
        let label = UILabel()
        label.textAlignment = .center
        label.font = UIFont.PFSC_R(ofSize: 13)
        label.textColor = UIColor.brandGrayColor8
        return label
    }()
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        backgroundColor = UIColor.brandGrayColor1
        addSubview(iconView)
        addSubview(contentLabel)
        
        contentLabel.snp.makeConstraints { make in
            make.bottom.equalToSuperview().offset(-14)
            make.left.right.equalToSuperview()
        }
        iconView.snp.makeConstraints { make in
            make.bottom.equalTo(contentLabel.snp.top).offset(-6)
            make.width.height.equalTo(24)
            make.centerX.equalToSuperview()
        }
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    func setup(text: String, iconString: String) {
        contentLabel.text = text
        iconView.image = UIImage(named: iconString)
    }
}

// 按钮点击回调协议
protocol YHCircleHeaderButtonsViewDelegate: AnyObject {
    func headerButtonsView(_ view: YHCircleHeaderButtonsView, didTapButtonAt index: Int)
}

class YHCircleHeaderButtonsView: UIView {
    
    weak var delegate: YHCircleHeaderButtonsViewDelegate?
    
    // 按钮数据模型
    private let buttonData: [(title: String, imageName: String)] = [
        ("找客户", "icon_find_customer"),
        ("找服务", "icon_find_service"), 
        ("拓人脉", "icon_expand_network"),
        ("办活动", "icon_organize_event")
    ]
    
    // 按钮数组
    private var buttons: [UIControl] = []
    
    // 容器栈视图
    private lazy var stackView: UIStackView = {
        let stack = UIStackView()
        stack.axis = .horizontal
        stack.distribution = .fillEqually
        stack.alignment = .fill
        stack.spacing = 10
        stack.translatesAutoresizingMaskIntoConstraints = false
        return stack
    }()
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        setupUI()
    }
    
    required init?(coder: NSCoder) {
        super.init(coder: coder)
        setupUI()
    }
    
    private func setupUI() {
        backgroundColor = .white
        
        addSubview(stackView)
        
        // 设置约束
        stackView.snp.makeConstraints { make in
            make.left.right.equalToSuperview().inset(20)
            make.top.equalToSuperview().offset(16)
            make.bottom.equalToSuperview().offset(-20)
        }
        
        // 创建按钮
        createButtons()
    }
    
    private func createButtons() {
        for (index, data) in buttonData.enumerated() {
            let button = createButton(title: data.title, imageName: data.imageName, tag: index)
            buttons.append(button)
            stackView.addArrangedSubview(button)
        }
    }
    
    private func createButton(title: String, imageName: String, tag: Int) -> UIControl {
        let button = YHHeaderButtonItemView()
        button.setup(text: title, iconString: imageName)
        button.tag = tag
        // 添加点击事件
        button.addTarget(self, action: #selector(buttonTapped(_:)), for: .touchUpInside)
        return button
    }
    
    @objc private func buttonTapped(_ sender: UIControl) {
        // 调用代理方法
        delegate?.headerButtonsView(self, didTapButtonAt: sender.tag)
    }
}

// MARK: - UICollectionReusableView 版本（用于 CollectionView Header）
class YHCircleHeaderReusableView: UICollectionReusableView {
    static let reuseIdentifier = "YHCircleHeaderReusableView"
    
    weak var delegate: YHCircleHeaderButtonsViewDelegate?
    
    private lazy var headerButtonsView: YHCircleHeaderButtonsView = {
        let view = YHCircleHeaderButtonsView()
        return view
    }()
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        setupUI()
    }
    
    required init?(coder: NSCoder) {
        super.init(coder: coder)
        setupUI()
    }
    
    private func setupUI() {
        backgroundColor = .white
        addSubview(headerButtonsView)
        
        headerButtonsView.snp.makeConstraints { make in
            make.edges.equalToSuperview()
        }
        
        headerButtonsView.delegate = self
    }
}

// MARK: - YHCircleHeaderButtonsViewDelegate
extension YHCircleHeaderReusableView: YHCircleHeaderButtonsViewDelegate {
    func headerButtonsView(_ view: YHCircleHeaderButtonsView, didTapButtonAt index: Int) {
        delegate?.headerButtonsView(view, didTapButtonAt: index)
    }
}
