//
//  YHIncomeInputMoneyCell.swift
//  galaxy
//
//  Created by alexzzw on 2024/11/15.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

class YHIncomeInputMoneyCell: YHResignDocumentCell {
    private let maxCount = 12

    static let cellReuseIdentifier = "YHIncomeInputMoneyCell"
    
    var didBeginEditing: ((UITextField) -> Void)?
    var didEndEditing: ((UITextField) -> Void)?
    
    private lazy var dotIcon: UIImageView = {
        let imageView = UIImageView(image: UIImage(named: "red_dot_6"))
        return imageView
    }()
    
    private lazy var infoTitleLabel: UILabel = {
        let label = UILabel()
        label.font = .PFSC_R(ofSize: 14)
        label.textColor = UIColor.mainTextColor
        label.numberOfLines = 0
        label.lineBreakMode = .byCharWrapping
        return label
    }()
    
    private lazy var infoDetailLabel: UILabel = {
        let label = UILabel()
        label.font = .PFSC_R(ofSize: 14)
        label.textColor = UIColor(hexString: "#6D788A")
        return label
    }()
    
    private lazy var inputTextField: UITextField = {
        let textField = UITextField(frame: CGRect(x: 0, y: 0, width: KScreenWidth, height: 40))
        textField.textColor = .mainTextColor
        textField.font = .PFSC_M(ofSize: 14)
        textField.placeholder = "请输入人民币金额"
        textField.borderStyle = .none
        textField.delegate = self
        textField.keyboardType = .numberPad
        return textField
    }()
    
    private lazy var bottomLineView: UIView = {
        let view = UIView()
        view.backgroundColor = UIColor(hexString: "#E9ECF0")
        return view
    }()
    
    private lazy var infoFailLabel: UILabel = {
        let label = UILabel()
        label.font = .PFSC_R(ofSize: 12)
        label.textColor = .failColor
        label.text = "请输入"
        return label
    }()
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        setupUI()
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    func setupCellInfo(title: String, detail: String?, inputValue: String?, showBottomLine: Bool = true, showStarIcon: Bool = false, needShowFailButton: Bool) {
        infoTitleLabel.text = title
        bottomLineView.isHidden = !showBottomLine
        inputTextField.text = inputValue
        dotIcon.isHidden = !showStarIcon
        if showStarIcon {
            infoTitleLabel.snp.remakeConstraints { make in
                make.top.equalToSuperview().offset(16)
                make.left.equalTo(dotIcon.snp.right).offset(2)
                make.bottom.lessThanOrEqualToSuperview().offset(-16)
                make.width.lessThanOrEqualTo(102)
            }
        } else {
            infoTitleLabel.snp.remakeConstraints { make in
                make.top.equalToSuperview().offset(16)
                make.left.equalToSuperview().offset(18)
                make.bottom.lessThanOrEqualToSuperview().offset(-16)
                make.width.lessThanOrEqualTo(102)
            }
        }
        infoDetailLabel.text = detail
        if let detail = detail, detail.count > 0 {
            infoDetailLabel.snp.remakeConstraints { make in
                make.top.equalTo(inputTextField.snp.bottom).offset(4)
                make.bottom.lessThanOrEqualToSuperview().offset(-16)
                make.left.equalToSuperview().offset(147)
                make.right.lessThanOrEqualToSuperview().offset(-18)
            }
        } else {
            infoDetailLabel.snp.remakeConstraints { make in
                make.top.equalTo(inputTextField.snp.bottom).offset(0)
                make.bottom.lessThanOrEqualToSuperview().offset(-16)
                make.left.equalToSuperview().offset(147)
                make.right.lessThanOrEqualToSuperview().offset(-18)
            }
        }
        
        infoFailLabel.isHidden = !needShowFailButton
        infoFailLabel.snp.remakeConstraints { make in
            if needShowFailButton {
                make.top.equalTo(infoTitleLabel.snp.bottom).offset(6)
            }
            make.left.equalToSuperview().offset(18)
            make.right.lessThanOrEqualToSuperview().offset(-18)
            make.bottom.lessThanOrEqualToSuperview().offset(-16)
        }
        
    }
}

extension YHIncomeInputMoneyCell {
    
    private func setupUI() {
        updateCellCorner(.mid)
        subContainerView.addSubview(dotIcon)
        subContainerView.addSubview(infoTitleLabel)
        subContainerView.addSubview(inputTextField)
        subContainerView.addSubview(infoDetailLabel)
        subContainerView.addSubview(bottomLineView)
        subContainerView.addSubview(infoFailLabel)
        
        dotIcon.setContentCompressionResistancePriority(.required, for: .horizontal)
        infoTitleLabel.setContentCompressionResistancePriority(.defaultHigh, for: .horizontal)
        infoDetailLabel.setContentCompressionResistancePriority(.defaultLow, for: .horizontal)
        inputTextField.setContentCompressionResistancePriority(.defaultLow, for: .horizontal)
        
        dotIcon.snp.makeConstraints { make in
            make.centerY.equalTo(infoTitleLabel)
            make.left.equalToSuperview().offset(18)
            make.width.height.equalTo(6)
        }
        
        infoTitleLabel.snp.makeConstraints { make in
            make.top.equalToSuperview().offset(16)
            make.left.equalTo(dotIcon.snp.right).offset(2)
            make.bottom.lessThanOrEqualToSuperview().offset(-16)
            make.width.lessThanOrEqualTo(102)
        }
        
        inputTextField.snp.makeConstraints { make in
            make.top.equalToSuperview().offset(16)
            make.left.equalToSuperview().offset(147)
            make.right.lessThanOrEqualToSuperview().offset(-18)
        }
        
        infoDetailLabel.snp.makeConstraints { make in
            make.top.equalTo(inputTextField.snp.bottom).offset(4)
            make.bottom.equalToSuperview().offset(-16)
            make.left.equalToSuperview().offset(147)
            make.right.lessThanOrEqualToSuperview().offset(-18)
        }
        infoFailLabel.isHidden = true
        infoFailLabel.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(18)
            make.right.lessThanOrEqualToSuperview().offset(-18)
            make.bottom.equalToSuperview().offset(-16)
        }
        
        bottomLineView.snp.makeConstraints { make in
            make.bottom.equalToSuperview()
            make.left.equalToSuperview().offset(18)
            make.right.equalToSuperview().offset(-18)
            make.height.equalTo(0.5)
        }
    }
}

extension YHIncomeInputMoneyCell: UITextFieldDelegate {
    // MARK: - UITextFieldDelegate
    
    func textFieldDidBeginEditing(_ textField: UITextField) {
        if textField == inputTextField {
            didBeginEditing?(textField)
        }
    }
    
    func textFieldShouldReturn(_ textField: UITextField) -> Bool {
        textField.endEditing(false)
        return true
    }
    
    func textFieldDidEndEditing(_ textField: UITextField) {
        if textField == inputTextField {
            didEndEditing?(textField)
        }
    }
    
    func textField(_ textField: UITextField, shouldChangeCharactersIn range: NSRange, replacementString string: String) -> Bool {
        if textField == inputTextField {
            /// 输入删除
            if string == "" {
                return true
            }
            /// 禁止拷贝非数字
            let number = CharacterSet(charactersIn: "0123456789")
            if (string as NSString).rangeOfCharacter(from: number.inverted).location != NSNotFound {
                return false
            }
            let count = (textField.text?.utf16.count ?? 0) - range.length + string.utf16.count
            if count > maxCount {
                return false
            }
            return true
        }
        return true
    }
}
