//
//  YHCertificateSearchViewController.swift
//  galaxy
//
//  Created by edy on 2024/3/18.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

class YHCertificateSearchViewController: YHBaseViewController {

    var items:[YHSupplementInfo] = []
    var viewModel: YHCertificateViewModel = YHCertificateViewModel()
    var orderId : Int?
    var categoryInfo: YHCategoryData?

    lazy var tableView: UITableView = {
        
        let tableView = UITableView(frame:.zero, style:.grouped)

        if #available(iOS 11.0, *) {
            tableView.contentInsetAdjustmentBehavior = .never
        }
        tableView.estimatedSectionHeaderHeight = 14.0
        tableView.estimatedSectionFooterHeight = 1.0
        tableView.showsVerticalScrollIndicator = false
        tableView.separatorStyle = .none
        tableView.delegate = self
        tableView.dataSource = self
        tableView.backgroundColor = UIColor(hexString:"#F8F8F8")
        tableView.register(UITableViewCell.self, forCellReuseIdentifier: "UITableViewCell")
        tableView.register(YHCertificateInfoCell.self, forCellReuseIdentifier: YHCertificateInfoCell.cellReuseIdentifier)
        tableView.backgroundView = emptyDataTipsView
        return tableView
    }()
    
    lazy var previewFileTool:YHFilePreviewTool = {
       let tool = YHFilePreviewTool(targetVC: self)
       return tool
    }()
    
    lazy var searchView:UIView = {
        let view = UIView()
        view.backgroundColor = .white
        view.addSubview(searchBar)
        searchBar.snp.makeConstraints { make in
            make.centerY.equalToSuperview()
            make.left.equalToSuperview().offset(16)
            make.size.equalTo(CGSize(width: KScreenWidth-32.0, height:YHCertificateSearchBar.height))
        }
        return view
    }()
    
    lazy var searchBar: YHCertificateSearchBar = {
        let bar = YHCertificateSearchBar(frame:CGRect(x: 0, y: 0, width:KScreenWidth-32, height:YHCertificateSearchBar.height))
        bar.searchBlock = {
            [weak self] text in
            guard let self = self else { return }
            requestList(keyWord: text)
        }
        bar.textChange = {
            [weak self] text in
            guard let self = self else { return }
            if isEmptyString(text) {
                requestList(keyWord: "")
            }
        }
        return bar
    }()
    
    lazy var emptyDataTipsView:UIView = {
        let view = UIView(frame: CGRect(x: 0, y: 0, width: KScreenWidth, height: 124))
        view.isHidden = false
        
        let imgView = UIImageView(image: UIImage(named: "service_center_no_data"))
        imgView.contentMode = .scaleAspectFill
        view.addSubview(imgView)
        
        let label = UILabel()
        label.textColor = UIColor.labelTextColor2
        label.font = UIFont.PFSC_R(ofSize: 14)
        label.textAlignment = .center
        label.text = "没有找到相关资料哦~".local
        view.addSubview(label)
        
        imgView.snp.makeConstraints { make in
            make.top.equalToSuperview().offset(130)
            make.width.height.equalTo(92)
            make.centerX.equalTo(view)
        }
        
        label.snp.makeConstraints { make in
            make.top.equalTo(imgView.snp.bottom).offset(12)
            make.centerX.equalTo(view)
        }
        return view
    }()
    
    override func viewDidLoad() {
        super.viewDidLoad()
        self.view.backgroundColor = .white
        self.gk_navTitle = "资料搜索".local
        createUI()
        requestList(keyWord:"")
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        if let text = searchBar.textField.text, !text.isEmpty {
            self.requestList(keyWord: text)
        }
    }
    
    func createUI() {
        view.addSubview(searchView)
        view.addSubview(tableView)
        searchView.snp.makeConstraints { make in
            make.left.right.equalToSuperview()
            make.top.equalToSuperview().offset(k_Height_NavigationtBarAndStatuBar)
            make.height.equalTo(32+YHCertificateSearchBar.height)
        }
        tableView.snp.makeConstraints { make in
            make.left.right.bottom.equalToSuperview()
            make.top.equalTo(searchView.snp.bottom)
        }
    }
}

extension YHCertificateSearchViewController: UITableViewDelegate, UITableViewDataSource {
    
    func numberOfSections(in tableView: UITableView) -> Int {
        return items.count
    }

    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return 1
    }
    
    func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        
        let cell = tableView.dequeueReusableCell(withIdentifier: YHCertificateInfoCell.cellReuseIdentifier, for: indexPath) as! YHCertificateInfoCell
        
        if 0 <= indexPath.section && indexPath.section < items.count {
            let item = items[indexPath.section]
            cell.updateModel(item)
            cell.templateBlock = {
                [weak self] in
                guard let self = self else { return }
                let view = YHCertificateTemplateSheetView.templateView(item.templateInfo)
                view.scanTemplateBlock = {
                    [weak self] path in
                    guard let self = self else { return }
                    if let url = path?.url, !url.isEmpty {
                        print(url)
                        self.previewFileTool.openXLSXRemoteFile(urlString:url, fileName: "预览")
                    }
                }
                view.show()
            }
        }
        return cell
    }
    
    func tableView(_ tableView: UITableView, heightForRowAt indexPath: IndexPath) -> CGFloat {
        return UITableView.automaticDimension
    }
    
    func tableView(_ tableView: UITableView, didSelectRowAt indexPath: IndexPath) {
       
        if 0 <= indexPath.section && indexPath.section < items.count {
            let item = items[indexPath.section]
            
            if item.checkStatus == YHCertificateUploadStatus.preUpload.rawValue { // 待上传
                showUploadSheetView(item)
                return
            }
            // 非待上传的进入详情
            let vc = YHCertificateUploadContentListVC()
            vc.orderId = orderId
            vc.categoryInfo = categoryInfo
            vc.supplementInfo = item
            self.navigationController?.pushViewController(vc)
        }
    }

    private func tableView(_ tableView: UITableView, viewForHeaderInSection section: Int) -> CGFloat {
        return 14.0
    }
    
    private func tableView(_ tableView: UITableView, viewForFooterInSection section: Int) -> CGFloat {
        return 1.0
    }
    
    func tableView(_ tableView: UITableView, viewForFooterInSection section: Int) -> UIView? {
        
        return UIView()
    }
    
    func tableView(_ tableView: UITableView, viewForHeaderInSection section: Int) -> UIView? {
        
        return UIView()
    }
}

extension YHCertificateSearchViewController {
    
    func requestList(keyWord:String?) {
        
        /*
         // 状态
         "status_all": "",
         // 类型
         "template_cate_id": "",
         // 搜索
         "keyword": "",
         */
        let keywords = keyWord ?? ""
        let params:[String:Any] = [
                                    "node": "1",
                                    "state": "1",
                                    "supplement_ids": [],
                                    "order_id" : orderId ?? 0,
                                    "keyword": keywords]
        
        print(params)
        // state: 3:待上传 2:已驳回 1:已完成 0:审核中
        viewModel.getCertificateList(params: params) {
            
            [weak self] listInfo, error in
            guard let self = self else { return }
            
            items.removeAll()
            if let listInfo = listInfo {
                categoryInfo = listInfo.categoryData
                let arr = listInfo.supplementList
                items.append(contentsOf: arr)
            }
            emptyDataTipsView.isHidden = items.count > 0
            self.tableView.reloadData()
            
            if let errorMsg = error?.errorMsg, errorMsg.count > 0 {
                YHHUD.flash(message: errorMsg, duration:3)
            }
        }
    }
    
    func uploadImage(_ img: UIImage, imageName:String, supplementInfo:YHSupplementInfo, callBack:((Bool)->())?)  {
        
        // 先OSS上传得到URL
        self.viewModel.uploadImage(img) {
            [weak self] url, error in
            guard let self = self else { return }
            if let url = url, !url.isEmpty {
                var suffix = "jpg"
                let suffixName = url.pathExtension.lowercased()
                if !suffixName.isEmpty {
                    suffix = suffixName
                }
                
                // 再调用业务接口
                let model = YHContentUrl()
                model.name = imageName
                model.url = url
                model.suffix = suffix
                self.viewModel.updateCertificateItem(supplementsId:supplementInfo.id, urlModel:model, operation: "add", rename: "") {
                    success, error in
                    self.requestList(keyWord: self.searchBar.textField.text)
                    callBack?(success)
                    
                }
                return
            }
            callBack?(false)
        }
    }
    
    func uploadFile(_ fileUrl:URL, supplementInfo:YHSupplementInfo, successBlock:(()->())?)  {
        if let fileData = try? Data(contentsOf: fileUrl) {
            // 将文件数据上传到服务器
            let size = String(format: "%.2fM", Double(fileData.count)/(1024.0 * 1024.0))
            print("\(size)")
            // 先OSS上传得到URL
            YHHUD.show(.progress(message: "上传中..."))
            self.viewModel.uploadFile(fileUrl.absoluteString) {
                [weak self] successUrl, error in
                guard let self = self else { return }
                YHHUD.hide()
                // 再调用业务接口
                if let successUrl = successUrl, !successUrl.isEmpty {
                    let fileName = successUrl.lastPathComponent
                    let suffixName = successUrl.pathExtension.lowercased()
                    let model = YHContentUrl()
                    model.name = fileName
                    model.url = successUrl
                    model.suffix = suffixName
                    YHHUD.show(.progress(message: "上传中..."))
                    self.viewModel.updateCertificateItem(supplementsId:supplementInfo.id, urlModel:model, operation: "add", rename: "") {
                        success, error in
                        YHHUD.hide()
                        if success {
                            YHHUD.flash(message: "上传成功")
                        } else {
                            var msg = "上传失败"
                            if let errorMsg = error?.errorMsg, !errorMsg.isEmpty {
                                msg = errorMsg
                            }
                            YHHUD.flash(message:msg)
                        }
                        self.requestList(keyWord: self.searchBar.textField.text)
                        if success {
                            successBlock?()
                        }
                    }
                    return
                }
                var msg = "上传图片失败"
                if let errorMsg = error?.errorMsg, !errorMsg.isEmpty {
                    msg = errorMsg
                }
                YHHUD.flash(message: msg)

            }
        } else {
            YHHUD.flash(message: "识别文件失败")
        }
    }
    
    func showUploadSheetView(_ supplementInfo: YHSupplementInfo) {
                
        let sheetView = YHCertificateUploadSheetView.sheetView()
        sheetView.maxSelectCount = 9
        sheetView.uploadFilesBlock = {
            [weak self] fileUrl in
            guard let self = self else { return }
            print(fileUrl)
             uploadFile(fileUrl,  supplementInfo: supplementInfo) {
                 // 待上传的进入详情
                 DispatchQueue.main.async {
                     let vc = YHCertificateUploadContentListVC()
                     vc.orderId = self.orderId
                     vc.categoryInfo = self.categoryInfo
                     vc.supplementInfo = supplementInfo
                     self.navigationController?.pushViewController(vc)
                 }
             }
        }
        
        sheetView.uploadImageBlock = {
            [weak self] imgArr in
            guard let self = self else { return }
            var successArr:[YHSelectImageItem] = []
            var failArr:[YHSelectImageItem] = []
            let group = DispatchGroup()

            YHHUD.show(.progress(message: "上传中..."))
            for item in imgArr {
                if let image = item.data {
                    group.enter()
                    uploadImage(image, imageName:item.name, supplementInfo: supplementInfo) {
                        success in
                        if success {
                            successArr.append(item)
                        } else {
                            failArr.append(item)
                        }
                        group.leave()
                    }
                }
            }
            
            // 等待所有任务完成
            group.notify(queue: .main) {
                // 所有任务完成，回到主线程继续执行
                print("所有任务完成")
                YHHUD.hide()
                if imgArr.count == successArr.count {
                    YHHUD.flash(message: "上传成功")
                    
                } else {
                    YHHUD.flash(message: "\(successArr.count)张照片上传成功\n\(failArr.count)张照片上传失败")
                }
                
                if successArr.count > 0 {
                    let vc = YHCertificateUploadContentListVC()
                    vc.orderId = self.orderId
                    vc.categoryInfo = self.categoryInfo
                    vc.supplementInfo = supplementInfo
                    self.navigationController?.pushViewController(vc)
                }
            }
        }
        sheetView.show()
    }
}
