//
//  YHGCVisaProgramPopVC.swift
//  galaxy
//
//  Created by alexzzw on 2024/11/11.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit
import AttributedString

class YHGCVisaProgramPopVC: YHBaseViewController {
    
    private lazy var topProgramIcon: UIImageView = {
        let view = UIImageView()
        return view
    }()
    
    private lazy var sureButton: UIButton = {
        let button = UIButton(type: .custom)
        button.titleLabel?.font = UIFont.PFSC_M(ofSize: 16)
        button.setTitle("我已知悉，确认", for: .normal)
        button.setTitle("我已知悉，确认", for: .highlighted)
        button.setTitleColor( UIColor(hex: 0xffffff), for: .normal)
        button.setTitleColor( UIColor(hex: 0xffffff), for: .highlighted)
        button.addTarget(self, action: #selector(clickSureButton), for: .touchUpInside)
        button.layer.cornerRadius = kCornerRadius3
        button.clipsToBounds = true
        button.backgroundColor = .brandMainColor
        return button
    }()
    
    private lazy var infoDetailLabel: UILabel = {
        let label = UILabel()
        label.textColor = .mainTextColor
        label.font = .PFSC_R(ofSize: 14)
        label.lineBreakMode = .byCharWrapping
        label.numberOfLines = 0
        return label
    }()
    
    private lazy var subContainerView: YHBaseDynamicCornerRadiusView = {
        let view = YHBaseDynamicCornerRadiusView(cornerRadius: 6, corner: .allCorners)
        view.backgroundColor = .contentBkgColor
        return view
    }()
    
    private lazy var infoMarkLabel: UILabel = {
        let label = UILabel()
        label.textColor = UIColor(hexString: "#6D788A")
        label.font = .PFSC_R(ofSize: 14)
        label.lineBreakMode = .byCharWrapping
        label.numberOfLines = 0
        return label
    }()
    
    var sureButtonEvent: (() -> Void)?
    
    private let type: YHGCApplicationType
    init(type: YHGCApplicationType) {
        self.type = type
        super.init(nibName: nil, bundle: nil)
    }
    
    @MainActor required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    override func viewDidLoad() {
        super.viewDidLoad()
        setupUI()
    }

}

extension YHGCVisaProgramPopVC {
    
    @objc private func clickSureButton() {
        sureButtonEvent?()
    }
    
    private func setupUI() {
        gk_navBarAlpha = 0
        gk_navigationBar.isHidden = true
        view.backgroundColor = UIColor.white
        view.addSubview(topProgramIcon)
        view.addSubview(sureButton)
        view.addSubview(infoDetailLabel)
        view.addSubview(subContainerView)
        subContainerView.addSubview(infoMarkLabel)
        
        let ratio = 124.0 / 375.0
        topProgramIcon.snp.makeConstraints { make in
            make.top.left.right.equalToSuperview()
            make.height.equalTo(topProgramIcon.snp.width).multipliedBy(ratio)
        }
        
        let widthRatio = KScreenWidth / 375.0
        
        infoDetailLabel.snp.makeConstraints { make in
            make.top.equalToSuperview().offset(124.0 * widthRatio)
            make.left.equalToSuperview().offset(20 * widthRatio)
            make.right.equalToSuperview().offset(-20 * widthRatio)
        }
        
        subContainerView.snp.makeConstraints { make in
            make.top.equalTo(infoDetailLabel.snp.bottom).offset(20.0 * widthRatio)
            make.left.equalToSuperview().offset(20 * widthRatio)
            make.right.equalToSuperview().offset(-20 * widthRatio)
        }
        
        infoMarkLabel.snp.makeConstraints { make in
            make.top.equalToSuperview().offset(16.0 * widthRatio)
            make.left.equalToSuperview().offset(16 * widthRatio)
            make.right.equalToSuperview().offset(-16 * widthRatio)
            make.bottom.equalToSuperview().offset(-16.0 * widthRatio)
        }
        
        sureButton.snp.makeConstraints { make in
            make.top.greaterThanOrEqualTo(subContainerView.snp.bottom).offset(10)
            make.left.equalToSuperview().offset(kMargin * widthRatio)
            make.right.equalToSuperview().offset(-kMargin * widthRatio)
            make.bottom.equalTo(view.safeAreaLayoutGuide.snp.bottom).offset(-10 * widthRatio)
            make.height.equalTo(48)
        }
        
        topProgramIcon.image = UIImage(named: type.programIconTitle())
        infoDetailLabel.text = type.detailText()
        infoMarkLabel.attributed.text = type.attributedTips()
        
        preferredContentSize = CGSize(width: KScreenWidth, height: 452 * widthRatio)
    }
}

private extension YHGCApplicationType {
    func programIconTitle() -> String {
        switch self {
        case .typeA:
            return "gc_visa_program_a"
        case .typeB:
            return "gc_visa_program_b"
        case .typeC:
            return "gc_visa_program_c"
        }
    }
    
    func detailText() -> String {
        switch self {
        case .typeA:
            return "在紧接申请前一年，全年收入达港币250万元或以上(或等值外币) 的人士"
        case .typeB:
            return "4个指定榜单中的世界百强大学的学士学位毕业生(以入境处内置榜单为准)，且过去5年累积有不低于3年的工作经验"
        case .typeC:
            return "过去5年在4个指定榜单中的世界百强大学取得学士学位(以入境处内置榜单为准)，但累积工作经验不足3年"
        }
    }
    
    func attributedTips() -> ASAttributedString {
        switch self {
        case .typeA:
            let attr1: ASAttributedString = .init(string: "注：必须是", .font(UIFont.PFSC_R(ofSize: 14)), .foreground(UIColor(hexString: "#6D788A") ?? UIColor.gray))
            let attr2: ASAttributedString = .init(string: "课税的就业或业务收入", .font(UIFont.PFSC_B(ofSize: 14)), .foreground(UIColor.brandMainColor))
            let attr3: ASAttributedString = .init(string: "，享受税收优惠、免税的收入，不计算在内", .font(UIFont.PFSC_R(ofSize: 14)), .foreground(UIColor(hexString: "#6D788A") ?? UIColor.gray))
            return attr1 + attr2 + attr3
        case .typeB:
            let attr1: ASAttributedString = .init(string: "注：必须是", .font(UIFont.PFSC_R(ofSize: 14)), .foreground(UIColor(hexString: "#6D788A") ?? UIColor.gray))
            let attr2: ASAttributedString = .init(string: "紧接申请", .font(UIFont.PFSC_B(ofSize: 14)), .foreground(UIColor.brandMainColor))
            let attr3: ASAttributedString = .init(string: "的前五年，累积工作经验不低于三年", .font(UIFont.PFSC_R(ofSize: 14)), .foreground(UIColor(hexString: "#6D788A") ?? UIColor.gray))
            return attr1 + attr2 + attr3
        case .typeC:
            let attr1: ASAttributedString = .init(string: "注：", .font(UIFont.PFSC_R(ofSize: 14)), .foreground(UIColor(hexString: "#6D788A") ?? UIColor.gray))
            let attr2: ASAttributedString = .init(string: "不适用于", .font(UIFont.PFSC_B(ofSize: 14)), .foreground(UIColor.brandMainColor))
            let attr3: ASAttributedString = .init(string: "在港修读全日制经本地评审课程而获得学士学位的非本地生", .font(UIFont.PFSC_R(ofSize: 14)), .foreground(UIColor(hexString: "#6D788A") ?? UIColor.gray))
            return attr1 + attr2 + attr3
        }
    }
}
