//
//  YHMediaProtocolUtils.swift
//  galaxy
//
//  Created by alexzzw on 2024/11/24.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import Foundation

// MARK: - Protocols
protocol YHMediaPlayerViewDelegate: AnyObject {
    func didTapPlayButton()
    func didTapStopButton()
    func didSeekToPosition(_ position: Float)
    func didChangeQuality(_ quality: YHVideoQuality)
    func didToggleFullscreen()
}

// MARK: - Enums
enum YHVideoQuality: String, CaseIterable {
    case auto = "自动"
    case sd = "标清"
    case hd = "高清"
    case fhd = "超清"
}

enum YHPlayerStatus {
    case idle
    case playing
    case paused
    case stopped
    case error(String)
}

// MARK: - Data Models
struct YHLiveRoomInfo {
    let roomId: String
    let anchorId: String
    let anchorName: String
    let streamUrl: String
    let title: String
    let coverUrl: String?
    var currentQuality: YHVideoQuality = .auto
    var viewerCount: Int = 0
    var likeCount: Int = 0
    var status: YHLiveRoomStatus = .idle
}

enum YHLiveRoomStatus {
    case idle
    case living
    case ended
    case error(String)
}

struct YHLiveMessage {
    let id: String = UUID().uuidString
    let type: YHLiveMessageType
    let sender: YHLiveUser
    let content: String
    let timestamp: Date
    
    static func createNormalMessage(sender: YHLiveUser, content: String) -> YHLiveMessage {
        return YHLiveMessage(type: .normal, sender: sender, content: content, timestamp: Date())
    }
    
    static func createSystemMessage(_ content: String) -> YHLiveMessage {
        let system = YHLiveUser(userId: "system", nickname: "系统", level: 0, isAdmin: true)
        return YHLiveMessage(type: .system, sender: system, content: content, timestamp: Date())
    }
}

enum YHLiveMessageType {
    case normal    // 普通消息
    case system    // 系统消息
    case gift      // 礼物消息
    case enter     // 进入消息
    case like      // 点赞消息
}

struct YHLiveUser {
    let userId: String
    let nickname: String
    let level: Int
    let isAdmin: Bool
    var avatar: String?
}

// MARK: - Video Models
struct YHVideoInfo {
    let videoId: String
    let title: String
    let description: String?
    let coverUrl: String?
    let duration: Int          // 视频时长(秒)
    let createTime: Date       // 创建时间
    var playCount: Int         // 播放次数
    let category: YHVideoCategory
    let author: YHVideoAuthor
    let videoUrls: [YHVideoQualityURL]  // 不同清晰度的视频地址
    
    // 获取指定清晰度的视频URL
    func getVideoUrl(for quality: YHVideoQuality) -> String? {
        return videoUrls.first { $0.quality == quality }?.url
    }
    
    // 获取最佳清晰度(根据网络情况)
    func getBestQualityUrl(for networkType: YHNetworkType) -> String? {
        switch networkType {
        case .wifi:
            return videoUrls.sorted { $0.quality.rawValue > $1.quality.rawValue }.first?.url
        case .cellular:
            return videoUrls.first { $0.quality == .sd }?.url
        case .none:
            return videoUrls.first { $0.quality == .sd }?.url
        }
    }
}

// 视频作者信息
struct YHVideoAuthor {
    let userId: String
    let nickname: String
    let avatarUrl: String?
    let description: String?
    var followerCount: Int
    var isFollowed: Bool
}

// 视频分类
enum YHVideoCategory: String {
    case entertainment = "娱乐"
    case sports = "体育"
    case news = "新闻"
    case education = "教育"
    case gaming = "游戏"
    case music = "音乐"
    case other = "其他"
}

// 不同清晰度的视频URL
struct YHVideoQualityURL {
    let quality: YHVideoQuality
    let url: String
    let bitrate: Int      // 比特率
    let resolution: YHVideoResolution
}

// 视频分辨率
struct YHVideoResolution {
    let width: Int
    let height: Int
    
    var description: String {
        return "\(width)x\(height)"
    }
}

// 网络类型
enum YHNetworkType {
    case wifi
    case cellular
    case none
}

// 视频播放相关的配置
struct YHVideoPlayConfig {
    var autoPlay: Bool = true          // 自动播放
    var loopPlay: Bool = false         // 循环播放
    var muteOnStart: Bool = false      // 开始时静音
    var preferredQuality: YHVideoQuality = .auto  // 首选清晰度
    var enableBackgroundPlay: Bool = false      // 允许后台播放
}

// 视频播放状态
enum YHVideoPlayState {
    case idle        // 空闲
    case loading     // 加载中
    case playing     // 播放中
    case paused      // 暂停
    case ended       // 结束
    case error(String)   // 错误
}

// 视频播放进度信息
struct YHVideoProgress {
    let currentTime: TimeInterval
    let duration: TimeInterval
    var progress: Float {
        return duration > 0 ? Float(currentTime / duration) : 0
    }
}
