//
//  YHPreviewMediaItem.swift
//  galaxy
//
//  Created by alexzzw on 2025/9/27.
//  Copyright © 2025 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

// 预览媒体项 - 专门用于预览的数据模型
struct YHPreviewMediaItem {
    let type: YHMediaType
    let source: YHMediaSource
    
    enum YHMediaSource {
        case remoteURL(String)           // 远程URL
        case localImage(UIImage)         // 本地图片
        case localVideoURL(URL)          // 本地视频文件URL
        
        var displayURL: String {
            switch self {
            case .remoteURL(let url):
                return url
            case .localImage:
                return "" // 本地图片不需要URL
            case .localVideoURL(let url):
                return url.absoluteString
            }
        }
    }
    
    // 从远程URL创建
    init(remoteURL: String, type: YHMediaType = .image) {
        self.type = type
        self.source = .remoteURL(remoteURL)
    }
    
    // 从本地图片创建
    init(localImage: UIImage) {
        self.type = .image
        self.source = .localImage(localImage)
    }
    
    // 从本地视频URL创建
    init(localVideoURL: URL) {
        self.type = .video
        self.source = .localVideoURL(localVideoURL)
    }
    
    // 从 YHSelectMediaItem 创建
    init(from selectMediaItem: YHSelectMediaItem) {
        self.type = selectMediaItem.type
        
        switch selectMediaItem.type {
        case .image:
            if let image = selectMediaItem.image {
                self.source = .localImage(image)
                
            } else if let url = selectMediaItem.imageURL {
                self.source = .remoteURL(url.absoluteString)
                
            } else {
                // 如果没有本地图片，使用占位符
                self.source = .localImage(UIImage(named: "global_default_image") ?? UIImage())
            }
        case .video:
            if let videoURL = selectMediaItem.videoURL {
                self.source = .localVideoURL(videoURL)
            } else {
                // 异常情况，创建一个空的远程URL
                self.source = .remoteURL("")
            }
        }
    }
}
