//
//  YHCircleViewController.swift
//  galaxy
//
//  Created by alexzzw on 2025/9/24.
//  Copyright © 2025 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit
import ESPullToRefresh
import JXSegmentedView

class YHCircleViewController: YHBaseViewController {
    
    lazy var viewModel: YHCircleViewModel = {
        let viewModel = YHCircleViewModel()
        return viewModel
    }()
    
    lazy var circleCollectView: UICollectionView = {
        let flowLayout = YHHomeCollectionLayout()
        flowLayout.minimumInteritemSpacing = 7.0
        flowLayout.minimumColumnSpacing = 7.0
        flowLayout.sectionInset = UIEdgeInsets(top: 0, left: 20, bottom: 0, right: 20)
        flowLayout.headerHeight = 112
        let collectionView = UICollectionView(frame: .zero, collectionViewLayout: flowLayout)
        collectionView.backgroundColor = .white
//        collectionView.register(YHCircleCollectionViewCell.self, forCellWithReuseIdentifier: YHCircleCollectionViewCell.cellReuseIdentifier)
        collectionView.register(YHHomeCollectionViewCell.self, forCellWithReuseIdentifier: YHHomeCollectionViewCell.cellReuseIdentifier)
        collectionView.register(YHCircleHeaderReusableView.self, forSupplementaryViewOfKind: CollectionViewWaterfallElementKindSectionHeader, withReuseIdentifier: YHCircleHeaderReusableView.reuseIdentifier)
        collectionView.delegate = self
        collectionView.dataSource = self
        collectionView.translatesAutoresizingMaskIntoConstraints = false
        collectionView.alwaysBounceVertical = true
        collectionView.showsHorizontalScrollIndicator = false
        collectionView.showsVerticalScrollIndicator = false
        return collectionView
    }()
    
    lazy var publishButton: UIButton = {
        let btn = UIButton(type: .custom)
        btn.backgroundColor = UIColor.brandGrayColor8
        btn.setImage(UIImage(named: "circle_publish_icon"), for: .normal)
        btn.setTitle("发布动态", for: .normal)
        btn.setTitleColor(UIColor.white, for: .normal)
        btn.titleLabel?.font = UIFont.PFSC_R(ofSize: 15)
        btn.layer.cornerRadius = 22
        btn.addTarget(self, action: #selector(publishButtonClicked), for: .touchUpInside)
        return btn
    }()

    override func viewDidLoad() {
        super.viewDidLoad()
        setupUI()
        addDefaultData()
        getData()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        if viewModel.arrCircleData?.isEmpty == true {
            getData()
        }
    }

    deinit {
        NotificationCenter.default.removeObserver(self)
    }
}

// MARK: - 私有方法
private extension YHCircleViewController {
    
    func setupUI() {
        gk_navigationBar.isHidden = true
        view.addSubview(circleCollectView)
        view.addSubview(publishButton)
        circleCollectView.snp.makeConstraints { make in
            make.top.left.right.bottom.equalToSuperview()
        }
        
        publishButton.snp.makeConstraints { make in
            make.width.equalTo(134)
            make.height.equalTo(44)
            make.centerX.equalToSuperview()
            make.bottom.equalTo(view.safeAreaLayoutGuide).offset(-30)
        }
        
        publishButton.iconInLeft(spacing: 6)
        
        circleCollectView.es.addYHPullToRefresh {
            self.getData()
        }
        
        circleCollectView.es.addInfiniteScrolling {
            self.loadMoreData()
        }
        
        circleCollectView.es.stopPullToRefresh()
        circleCollectView.es.stopLoadingMore()
    }
    
    func addDefaultData() {
        getCacheDataForCircleData()
    }
    
    func getCacheDataForCircleData() {
        let arrString = UserDefaults.standard.value(forKey: "circleFirstPageData")
        if let tmp = arrString as? String, let dic = tmp.toArray() {
            guard let result = [YHCircleListModel].deserialize(from: dic) else {
                return
            }
            
            if let arrData = result as? [YHHomeListModel] {
                for item in arrData {
                    item.calHeightParam()
                }
                
                self.viewModel.arrCircleData = arrData
                self.circleCollectView.reloadData()
            }
        }
    }
    
    func getData() {
        loadFirstData()
    }
    
    func loadFirstData() {
        DispatchQueue.global().async {
            self.viewModel.getCircleList(firstFlag: true) { [weak self] _, _ in
                guard let self = self else { return }
                
                if let arrData = self.viewModel.arrCircleData, arrData.count > 0 {
                    let arrString = arrData.toJSONString()
                    UserDefaults.standard.set(arrString, forKey: "circleFirstPageData")
                    UserDefaults.standard.synchronize()
                }
                
                DispatchQueue.main.async {
                    self.circleCollectView.es.stopPullToRefresh()
                    
                    if self.viewModel.hasMoreForCircle == false {
                        self.circleCollectView.es.noticeNoMoreData()
                        self.circleCollectView.footer?.alpha = 1
                    }
                    
                    self.circleCollectView.reloadData()
                }
            }
        }
    }
    
    func loadMoreData() {
        DispatchQueue.global().async {
            self.viewModel.getCircleList(firstFlag: false) { [weak self] _, _ in
                guard let self = self else { return }
                
                DispatchQueue.main.asyncAfter(wallDeadline: .now() + 0.35, execute: {
                    CATransaction.setDisableActions(true)
                    self.circleCollectView.reloadData()
                    CATransaction.commit()
                    
                    self.circleCollectView.es.stopLoadingMore()
                    if self.viewModel.hasMoreForCircle == false {
                        self.circleCollectView.es.noticeNoMoreData()
                        self.circleCollectView.footer?.alpha = 1
                    }
                })
            }
        }
    }
    
    @objc func publishButtonClicked() {
        if !YHLoginManager.shared.isLogin() {
            YHOneKeyLoginManager.shared.oneKeyLogin()
            return
        }
        
        let publishVC = YHCirclePublishViewController()
        publishVC.completion = { [weak self] in
            self?.getData()
        }
        let nav = YHNavigationController(rootViewController: publishVC)
        nav.modalPresentationStyle = .fullScreen
        present(nav, animated: true)
    }
    
    // MARK: - 头部按钮处理方法
    func handleHeaderButtonTap(at index: Int) {
        switch index {
        case 0: // 找客户
            handleFindCustomerAction()
        case 1: // 找服务
            handleFindServiceAction()
        case 2: // 拓人脉
            handleExpandNetworkAction()
        case 3: // 办活动
            handleOrganizeEventAction()
        default:
            break
        }
    }
    
    private func handleFindCustomerAction() {
        print("找客户按钮被点击")
        // TODO: 跳转到找客户页面
        // let findCustomerVC = YHFindCustomerViewController()
        // navigationController?.pushViewController(findCustomerVC, animated: true)
    }
    
    private func handleFindServiceAction() {
        print("找服务按钮被点击")
        // TODO: 跳转到找服务页面
        // let findServiceVC = YHFindServiceViewController()
        // navigationController?.pushViewController(findServiceVC, animated: true)
    }
    
    private func handleExpandNetworkAction() {
        print("拓人脉按钮被点击")
        // TODO: 跳转到拓人脉页面
        // let expandNetworkVC = YHExpandNetworkViewController()
        // navigationController?.pushViewController(expandNetworkVC, animated: true)
    }
    
    private func handleOrganizeEventAction() {
        print("办活动按钮被点击")
        // TODO: 跳转到办活动页面
        // let organizeEventVC = YHOrganizeEventViewController()
        // navigationController?.pushViewController(organizeEventVC, animated: true)
    }
}

// MARK: - UICollectionViewDelegate, UICollectionViewDataSource
extension YHCircleViewController: UICollectionViewDelegate, UICollectionViewDataSource {
    
    func collectionView(_ collectionView: UICollectionView, numberOfItemsInSection section: Int) -> Int {
        return self.viewModel.arrCircleData?.count ?? 0
    }
    
    func collectionView(_ collectionView: UICollectionView, cellForItemAt indexPath: IndexPath) -> UICollectionViewCell {
        guard let datas = self.viewModel.arrCircleData, datas.count > indexPath.row else {
            return UICollectionViewCell()
        }
        guard let cell = collectionView.dequeueReusableCell(withReuseIdentifier: YHHomeCollectionViewCell.cellReuseIdentifier, for: indexPath) as? YHHomeCollectionViewCell else {
            return UICollectionViewCell()
        }
        
        cell.listModel = datas[indexPath.row]
        
//        cell.clickHeartEvent = { [weak self] item in
//            self?.viewModel.toggleLike(postId: item.id) { success, _ in
//                if success {
//                    DispatchQueue.main.async {
//                        self?.circleCollectView.reloadItems(at: [indexPath])
//                    }
//                }
//            }
//        }
        
        return cell
    }
    
    // MARK: - Header View
    func collectionView(_ collectionView: UICollectionView, viewForSupplementaryElementOfKind kind: String, at indexPath: IndexPath) -> UICollectionReusableView {
        if kind == CollectionViewWaterfallElementKindSectionHeader, let headerView = collectionView.dequeueReusableSupplementaryView(ofKind: kind, withReuseIdentifier: YHCircleHeaderReusableView.reuseIdentifier, for: indexPath) as? YHCircleHeaderReusableView {
            
            headerView.delegate = self
            
            return headerView
        }
        
        return UICollectionReusableView()
    }
    
    func collectionView(_ collectionView: UICollectionView, didSelectItemAt indexPath: IndexPath) {
        guard let model = self.viewModel.arrCircleData?[indexPath.row] else { return }
        
        let detailVC = YHCircleDetailViewController()
//        detailVC.circleModel = model
//        detailVC.block = { item in
//            if let arr = self.viewModel.arrCircleData {
//                for (index, targetItem) in arr.enumerated() {
//                    if targetItem.id == item.id {
//                        targetItem.is_like = item.is_like
//                        targetItem.like_count = item.like_count
//                        self.circleCollectView.reloadItems(at: [IndexPath(item: index, section: 0)])
//                    }
//                }
//            }
//        }
        
        self.navigationController?.pushViewController(detailVC, animated: true)
    }
}

// MARK: - CollectionViewWaterfallLayoutDelegate
extension YHCircleViewController: CollectionViewWaterfallLayoutDelegate {
    
    func collectionView(_ collectionView: UICollectionView, layout: UICollectionViewLayout, sizeForItemAtIndexPath indexPath: NSIndexPath) -> CGSize {
        let count = self.viewModel.arrCircleData?.count ?? 0
        
        if indexPath.row < count {
            guard let model = self.viewModel.arrCircleData?[indexPath.row] else { return CGSize.zero }
            return CGSize(width: model.img_width_cell, height: model.img_height_cell)
        }
        
        let width = (KScreenWidth - 47) / 2
        return CGSize(width: width, height: 200.0)
    }
    
    func collectionView(_ collectionView: UICollectionView, layout collectionViewLayout: UICollectionViewLayout, referenceSizeForHeaderInSection section: Int) -> CGSize {
        return CGSize(width: collectionView.bounds.width, height: 110)
    }
}

// MARK: - UIScrollViewDelegate
extension YHCircleViewController {
    
    func scrollViewDidScroll(_ scrollView: UIScrollView) {
        let itemNumber = viewModel.preloadItemIndex
        let section = 0
        let itemIndex = itemNumber - 1
        let indexPathToCheck = IndexPath(item: itemIndex, section: section)
        
        if circleCollectView.indexPathsForVisibleItems.contains(indexPathToCheck) {
            loadMoreData()
        }
    }
}

// MARK: - JXSegmentedListContainerViewListDelegate
extension YHCircleViewController: JXSegmentedListContainerViewListDelegate {
    
    func listView() -> UIView {
        return view
    }
}

// MARK: - YHCircleHeaderButtonsViewDelegate
extension YHCircleViewController: YHCircleHeaderButtonsViewDelegate {
    
    func headerButtonsView(_ view: YHCircleHeaderButtonsView, didTapButtonAt index: Int) {
        handleHeaderButtonTap(at: index)
    }
}
