//
//  YHHUD.swift
//  PKHUD Demo
//
//  Created by ahao on 2018/3/20.
//  Copyright © 2018年 NSExceptional. All rights reserved.
//

import UIKit
import PKHUD

public enum YHHUDType {
    case success(message: String?)
    case error(message: String?)
    case warning(message: String?)
    case progress(message: String?)
    case image(image: UIImage?, message: String?)
    case rotatingImage(image: UIImage?, message: String?)
}

class YHHUD {
    // MARK: Properties
    public static var dimsBackground: Bool {
        get { return PKHUD.sharedHUD.dimsBackground }
        set { PKHUD.sharedHUD.dimsBackground = newValue }
    }
    
    public static var allowsInteraction: Bool {
        get { return PKHUD.sharedHUD.userInteractionOnUnderlyingViewsEnabled  }
        set { PKHUD.sharedHUD.userInteractionOnUnderlyingViewsEnabled = newValue }
    }
    
    public static var isVisible: Bool { return PKHUD.sharedHUD.isVisible }
    
    // MARK: Public methods, PKHUD based
    public static func show(_ content: YHHUDType, onView view: UIView? = nil) {
        PKHUD.sharedHUD.contentView = contentView(content)
        PKHUD.sharedHUD.show(onView: view)
    }
    
    public static func hide(_ completion: ((Bool) -> Void)? = nil) {
        PKHUD.sharedHUD.hide(animated: false, completion: completion)
    }
    
    public static func hide(animated: Bool, completion: ((Bool) -> Void)? = nil) {
        PKHUD.sharedHUD.hide(animated: animated, completion: completion)
    }
    
    public static func hide(afterDelay delay: TimeInterval, completion: ((Bool) -> Void)? = nil) {
        PKHUD.sharedHUD.hide(afterDelay: delay, completion: completion)
    }
    
    public static func hideFlashMessage() {

        let keyWindow = UIApplication.shared.windows.first {$0.isKeyWindow}

        if keyWindow != nil {
            while keyWindow!.viewWithTag(YHHUD.flashMessageTag) != nil {
                keyWindow!.viewWithTag(YHHUD.flashMessageTag)?.removeFromSuperview()
            }
        }
    }
    
    // MARK: Public methods, HUD based
    public static func flash(_ content: YHHUDType, onView view: UIView? = nil) {
		YHHUD.flash(content, onView: view, delay: 1, completion: nil)
    }
    
    public static func flash(_ content: YHHUDType, onView view: UIView? = nil, delay: TimeInterval, completion: ((Bool) -> Void)? = nil) {
        YHHUD.show(content, onView: view)
        YHHUD.hide(afterDelay: delay, completion: completion)
    }
    
    // MARK: Private methods
    static func contentView(_ content: YHHUDType) -> UIView {
        let todoView: UIView
        switch content {
        case let .progress(message):
            todoView = YHHUDProgressView(title: message)
        case let .success(message):
            todoView = YHHUDSuccessView(title: message)
        case let .error(message):
            todoView = YHHUDErrorView(title: message)
        case let .warning(message):
            todoView = YHHUDWariningView(title: message)
        case let .rotatingImage(image, message):
            todoView = YHHUDSquareBaseView(image: image, title: message)
        case let .image(image, message):
            todoView = YHHUDSquareBaseView(image: image, title: message)
        }
        return todoView
    }
}

extension YHHUD {
    static func setup() {
        PKHUD.sharedHUD.effect = nil
//        PKHUD.sharedHUD.gracePeriod = 0.5
    }
}

extension YHHUD {
    static let flashMessageTag = 61535
    static func flash(message: String,
                      textAligment: NSTextAlignment = .center,
                      dismissAfter: TimeInterval = 1.5,
                      duration: TimeInterval = 0.5,
                      isRemoveBefore: Bool = true,
                      insets: UIEdgeInsets = UIEdgeInsets(top: 0, left: 20, bottom: 120, right: 20),
                      contentInsets: UIEdgeInsets = UIEdgeInsets(top: 12, left: 21, bottom: 12, right: 21),
                      containerWindow: UIWindow? = nil,
                      walk: ((UILabel) -> Void)? = nil) {
        if isRemoveBefore {
            hideFlashMessage()
        }
        if message.isEmpty {
            return
        }
        let containerView = UIView()
        containerView.backgroundColor = UIColor.black.withAlphaComponent(0.7)
        containerView.layer.cornerRadius = 4
        containerView.alpha = 0
        containerView.tag = flashMessageTag
        let label = UILabel()
        label.text = message
        label.textAlignment = textAligment
        label.numberOfLines = 5
        label.textColor = UIColor.white
        label.font = UIFont.systemFont(ofSize: 14)
        containerView.addSubview(label)
        walk?(label)

        let keyW = UIApplication.shared.windows.first {$0.isKeyWindow}

        if let keyWindow = (containerWindow ?? keyW) {
            let labelWidth = keyWindow.bounds.width-insets.left-insets.right-contentInsets.left-contentInsets.right
            label.frame = CGRect(x: contentInsets.left,
                             y: contentInsets.top,
                             width: labelWidth,
                             height: 30)
            label.sizeToFit()
            let containerViewSize = CGSize(width: label.frame.width+contentInsets.left+contentInsets.right,
                                           height: label.frame.height+contentInsets.top+contentInsets.bottom)
            containerView.frame = CGRect(x: (keyWindow.bounds.width-containerViewSize.width)/2.0,
                                         y: (keyWindow.bounds.height-containerViewSize.height)/2.0,
                                         width: containerViewSize.width,
                                         height: containerViewSize.height)
            keyWindow.addSubview(containerView)
            UIView.animate(withDuration: duration, animations: {
                containerView.alpha = 1
            }, completion: { (_) in
                UIView.animate(withDuration: duration, delay: dismissAfter, options: [], animations: {
                    containerView.alpha = 0
                }, completion: { (_) in
                    containerView.removeFromSuperview()
                })
            })
        }
    }
}
