//
//  YHCustomSearchView.swift
//  galaxy
//
//  Created by alexzzw on 2025/9/24.
//  Copyright © 2025 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

protocol YHCustomSearchViewDelegate: AnyObject {
    func searchView(_ searchView: YHCustomSearchView, didSearchWithText text: String?)
    func searchViewDidBeginEditing(_ searchView: YHCustomSearchView)
    func searchViewDidEndEditing(_ searchView: YHCustomSearchView)
}

class YHCustomSearchView: UIView {
    
    weak var delegate: YHCustomSearchViewDelegate?
    
    // MARK: - UI Components
    
    private lazy var containerView: UIView = {
        let view = UIView()
        view.backgroundColor = .brandGrayColor1
        return view
    }()
    
    private lazy var searchIconImageView: UIImageView = {
        let imageView = UIImageView()
        imageView.image = UIImage(named: "resource_search")
        imageView.contentMode = .scaleAspectFit
        return imageView
    }()
    
    private lazy var searchTextField: UITextField = {
        let textField = UITextField()
        textField.font = UIFont.PFSC_M(ofSize: 14)
        textField.textColor = UIColor.brandGrayColor8
        textField.borderStyle = .none
        textField.delegate = self
        textField.addTarget(self, action: #selector(textFieldEditingChanged(_:)), for: .editingChanged)
        textField.addTarget(self, action: #selector(textFieldEditingDidBegin(_:)), for: .editingDidBegin)
        textField.addTarget(self, action: #selector(textFieldEditingDidEnd(_:)), for: .editingDidEnd)
        textField.returnKeyType = .search
        textField.clearButtonMode = .whileEditing
        textField.attributedPlaceholder = NSAttributedString.init(string: "搜索企业名称、企业供需", attributes: [.foregroundColor: UIColor.brandGrayColor6, .font: UIFont.PFSC_R(ofSize: 14)])
        return textField
    }()
    
    // MARK: - Initialization
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        setupUI()
    }
    
    required init?(coder: NSCoder) {
        super.init(coder: coder)
        setupUI()
    }
    
    // MARK: - Setup
    
    private func setupUI() {
        backgroundColor = .white
        
        addSubview(containerView)
        containerView.addSubview(searchIconImageView)
        containerView.addSubview(searchTextField)
        
        setupConstraints()
    }
    
    private func setupConstraints() {
        containerView.snp.makeConstraints { make in
            make.left.right.equalToSuperview().inset(20)
            make.centerY.equalToSuperview()
            make.height.equalTo(36)
        }
        
        searchIconImageView.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(12)
            make.centerY.equalToSuperview()
            make.width.height.equalTo(24)
        }
        
        searchTextField.snp.makeConstraints { make in
            make.left.equalTo(searchIconImageView.snp.right).offset(6)
            make.right.equalToSuperview().offset(-12)
            make.top.bottom.equalToSuperview()
        }
    }
    
    // MARK: - Public Methods
    
    var text: String? {
        get {
            return searchTextField.text
        }
        set {
            searchTextField.text = newValue
        }
    }
    
    var placeholder: String? {
        get {
            return searchTextField.placeholder
        }
        set {
            searchTextField.placeholder = newValue
        }
    }
    
    @discardableResult
    override func becomeFirstResponder() -> Bool {
        return searchTextField.becomeFirstResponder()
    }
    
    @discardableResult
    override func resignFirstResponder() -> Bool {
        return searchTextField.resignFirstResponder()
    }
    
    func clearText() {
        searchTextField.text = ""
        delegate?.searchView(self, didSearchWithText: "")
    }
    
    // MARK: - Actions
    
    @objc private func textFieldEditingChanged(_ textField: UITextField) {
        delegate?.searchView(self, didSearchWithText: textField.text)
    }
    
    @objc private func textFieldEditingDidBegin(_ textField: UITextField) {
        delegate?.searchViewDidBeginEditing(self)
    }
    
    @objc private func textFieldEditingDidEnd(_ textField: UITextField) {
        delegate?.searchViewDidEndEditing(self)
    }
}

// MARK: - UITextFieldDelegate
extension YHCustomSearchView: UITextFieldDelegate {
    
    func textFieldShouldReturn(_ textField: UITextField) -> Bool {
        textField.resignFirstResponder()
        delegate?.searchView(self, didSearchWithText: textField.text)
        return true
    }
}
