//
//  YHIncomeTypesCell.swift
//  galaxy
//
//  Created by alexzzw on 2024/11/15.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

class YHIncomeTypesCell: YHResignDocumentCell {
    static let cellReuseIdentifier = "YHIncomeTypesCell"
    
    var didClickTypeView: ((Int) -> Void)?
    private var typeViews: [YHIncomeTypeView] = []
    private var lastTypes: [String] = []

    private lazy var bottomLineView: UIView = {
        let view = UIView()
        view.backgroundColor = UIColor(hexString: "#E9ECF0")
        return view
    }()

    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        setupUI()
    }

    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }

    private func setupUI() {
        updateCellCorner(.mid)
        subContainerView.addSubview(bottomLineView)
        bottomLineView.snp.makeConstraints { make in
            make.bottom.equalToSuperview()
            make.left.equalToSuperview().offset(18)
            make.right.equalToSuperview().offset(-18)
            make.height.equalTo(0.5)
        }
    }

    func setupCellInfo(types: [String]) {
        guard lastTypes != types || types.count != typeViews.count else {
            return
        }
        lastTypes = types
        typeViews.forEach {
            $0.removeFromSuperview()
        }
        typeViews.removeAll()
        let count = types.count
        types.enumerated().forEach { [weak self] index, title in
            guard let self = self else {
                return
            }
            let typeView = YHIncomeTypeView()
            typeView.clipsToBounds = true
            typeView.layer.cornerRadius = 3
            typeView.setupInfo(title)
            typeView.closeButtonEvent = { [weak self] in
                guard let self = self else {
                    return
                }
                self.didClickTypeView?(index)
            }
            self.typeViews.append(typeView)
            self.subContainerView.addSubview(typeView)

            typeView.snp.makeConstraints { make in
                if index == 0 {
                    make.top.equalToSuperview()
                } else {
                    make.top.equalTo(self.typeViews[index - 1].snp.bottom).offset(8)
                }
                make.left.equalToSuperview().offset(18)
                make.right.lessThanOrEqualToSuperview().offset(-18)
                if index == count - 1 {
                    make.bottom.equalToSuperview().offset(-16)
                }
            }
        }
    }
}

private class YHIncomeTypeView: UIView {
    var closeButtonEvent: (() -> Void)?

    private lazy var infoDetailLabel: UILabel = {
        let label = UILabel()
        label.font = .PFSC_M(ofSize: 12)
        label.textColor = .mainTextColor
        label.numberOfLines = 0
        label.lineBreakMode = .byCharWrapping
        return label
    }()

    private lazy var closeButton: UIButton = {
        let button = UIButton(type: .custom)
        button.setImage(UIImage(named: "income_type_close"), for: .normal)
        button.addTarget(self, action: #selector(closeButtonClicked), for: .touchUpInside)
        return button
    }()

    private lazy var lineView: UIView = {
        let view = UIView()
        view.backgroundColor = .white
        return view
    }()

    override init(frame: CGRect) {
        super.init(frame: frame)
        setupUI()
    }

    @MainActor required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }

    @objc private func closeButtonClicked() {
        closeButtonEvent?()
    }

    func setupInfo(_ detail: String) {
        infoDetailLabel.text = detail
    }

    private func setupUI() {
        backgroundColor = UIColor(hexString: "#F5F6F8")
        addSubview(infoDetailLabel)
        addSubview(lineView)
        addSubview(closeButton)
        infoDetailLabel.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(10)
            make.top.equalToSuperview().offset(8)
            make.bottom.equalToSuperview().offset(-8)
        }
        lineView.snp.makeConstraints { make in
            make.left.equalTo(infoDetailLabel.snp.right).offset(8)
            make.height.equalTo(20)
            make.width.equalTo(1)
            make.centerY.equalToSuperview()
        }
        closeButton.snp.makeConstraints { make in
            make.right.equalToSuperview()
            make.left.equalTo(lineView.snp.right)
            make.width.height.equalTo(26)
            make.centerY.equalToSuperview()
        }
    }
}
