//
//  YHAICardItemView.swift
//  galaxy
//
//  Created by Dufet on 2024/12/3.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit
import SDWebImage

class YHAICardItemView: UIView {
    
    var updateBlock: (() -> Void)?
    var evaluationResultCallback: ((Dictionary<String, Any>) -> Void)?

    var cardModel = YHAIListItemModel() {
        didSet {
            cardTitleLabel.text = cardModel.title
            describeLabel.text = cardModel.description
            
            // 尝试从缓存中获取图片
            if let cachedImage = SDImageCache.shared.imageFromCache(forKey: cardModel.cover) {
                // 如果缓存中有图片，则直接使用
                cardImgView.image = cachedImage
                let ratio = self.getSizeRatio(img: cachedImage)
                self.cardImgView.snp.remakeConstraints { make in
                    make.left.equalTo(16)
                    make.right.equalTo(-16)
                    make.top.equalTo(self.lineView.snp.bottom).offset(16)
                    make.height.equalTo(self.cardImgView.snp.width).multipliedBy(ratio)
                    make.bottom.equalTo(self.cardTitleLabel.snp.top).offset(-10)
                }

            } else {
                cardImgView.sd_setImage(with: URL(string: cardModel.cover), placeholderImage: UIImage(named: "global_default_image"), options: [], completed: { (_, _, _, _) in
                    self.updateBlock?()
                })
            }
            self.setNeedsLayout()
            self.layoutIfNeeded()
        }
    }
    
    func getSizeRatio(img: UIImage?) -> CGFloat {
        let size = img?.size ?? CGSize(width: 100, height: 50)
        var ratio = 0.5
        if size.width != 0.0, size.height != 0.0 {
            ratio = size.height/size.width
        }
        return ratio
    }
        
    lazy var lineView: UIView = {
        let v = UIView()
        v.backgroundColor = .init(hex: 0xE9ECF0)
        return v
    }()
    
    lazy var cardImgView: UIImageView = {
        let v = UIImageView(image: UIImage(named: "global_default_image"))
        v.layer.cornerRadius = 6.0
        v.clipsToBounds = true
        return v
    }()
    
    lazy var cardTitleLabel: UILabel = {
        let lable = UILabel()
        lable.textColor = UIColor.mainTextColor
        lable.textAlignment = .left
        lable.font = UIFont.PFSC_B(ofSize: 16)
        lable.numberOfLines = 0
        lable.text = "卡片标题"
        return lable
    }()
    
    lazy var describeLabel: UILabel = {
        let lable = UILabel()
        lable.textColor = UIColor.init(hex: 0x8993A2)
        lable.textAlignment = .left
        lable.font = UIFont.PFSC_R(ofSize: 12)
        lable.numberOfLines = 0
        lable.text = "卡片描述文字"
        return lable
    }()
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        createUI()
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    func createUI() {
        
        let tap = UITapGestureRecognizer(target: self, action: #selector(didItemViewClicked))
        self.addGestureRecognizer(tap)
        
        self.addSubview(lineView)
        self.addSubview(cardImgView)
        self.addSubview(cardTitleLabel)
        self.addSubview(describeLabel)
        
        lineView.snp.makeConstraints { make in
            make.left.equalTo(16)
            make.right.equalTo(-16)
            make.top.equalTo(0)
            make.height.equalTo(0.5)
        }
        
        cardImgView.snp.makeConstraints { make in
            make.left.equalTo(16)
            make.right.equalTo(-16)
            make.top.equalTo(lineView.snp.bottom).offset(16)
            make.height.equalTo(cardImgView.snp.width).multipliedBy(0.5)
            make.bottom.equalTo(cardTitleLabel.snp.top).offset(-10)
        }
        
        cardTitleLabel.snp.makeConstraints { make in
            make.left.equalTo(16)
            make.right.equalTo(-16)
        }
        
        describeLabel.snp.makeConstraints { make in
            make.top.equalTo(cardTitleLabel.snp.bottom).offset(4)
            make.left.equalTo(16)
            make.right.equalTo(-16)
            make.bottom.equalTo(-16)
        }
    }
    
    @objc func didItemViewClicked() {
        
        UIApplication.shared.yhKeyWindow()?.endEditing(true)

        YHAIJumpPageTool.jumpPageWithType(mode: cardModel.redirectMode, path: cardModel.redirectPath) {
            dict in
            self.evaluationResultCallback?(dict)
        }
    }
}
