//
//  YHPrincipleEnvelopeCell.swift
//  galaxy
//
//  Created by edy on 2024/8/6.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

class YHPrincipleEnvelopeCell: UITableViewCell {
    
    static let cellReuseIdentifier = "YHPrincipleEnvelopeCell"
    var itemModel: YHPrincipleEnvelopeModel?
    var viewModel = YHPrincleViewModel()
    
    var expandClick: ((Bool) -> Void)?

    lazy var titleLabel: UILabel = {
        let label = UILabel()
        label.font = UIFont.PFSC_R(ofSize: 15)
        label.textColor = UIColor.mainTextColor
        label.text = "已获得港府甄选名额"
        return label
    }()
    
    lazy var iconImgView: UIImageView = {
        let view = UIImageView(image: UIImage(named: "custom_service_icon_right"))
        return view
    }()
    
    lazy var whiteView: UIView = {
        let view = UIView()
        view.backgroundColor = .white
        view.layer.cornerRadius = 6.0
        view.clipsToBounds = true
        view.isUserInteractionEnabled = true
        let gap = UITapGestureRecognizer(target: self, action: #selector(didWatchClicked))
        view.addGestureRecognizer(gap)
        return view
    }()
    
    lazy var expandBtn: UIButton = {
        let btn = UIButton()
        btn.setImage(UIImage(named: "custom_service_expand_0"), for: .normal)
        btn.addTarget(self, action: #selector(didExpandBtnClicked), for: .touchUpInside)
        btn.YH_clickEdgeInsets = UIEdgeInsets(top: 0, left: KScreenWidth, bottom: 20, right: 40)
        return btn
    }()
    
    lazy var envelopImgView: UIImageView = {
        let view = UIImageView(image: UIImage(named: "principle_envelope_unsee"))
        view.addSubview(self.nameLabel)
        nameLabel.snp.makeConstraints { make in
            make.top.equalTo(58.0)
            make.left.equalTo(20)
            make.right.equalTo(-20)
        }
        return view
    }()
    
    lazy var nameLabel: UILabel = {
        let label = UILabel()
        label.numberOfLines = 2
        label.textAlignment = .center
        label.font = UIFont.PFSC_M(ofSize: 18)
        label.textColor = UIColor(hex: 0xB75E24)
        return label
    }()
    
    lazy var previewFileTool: YHFilePreviewTool = {
        let tool = YHFilePreviewTool(targetVC: UIViewController.current)
       return tool
    }()
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        setupUI()
    }
    
    required init?(coder: NSCoder) {
        super.init(coder: coder)
    }
    
    @objc func didExpandBtnClicked() {
        guard let model = itemModel else { return }
        expandClick?(!model.isUnfold)
    }
    
    @objc func didWatchClicked() {
        guard let model = itemModel else { return }
        if !model.url.isEmpty {
            YHHUD.show(.progress(message: "加载中..."))
            self.viewModel.getRealUsefulUrl(model.url) {
                [weak self] resultUrl in
                YHHUD.hide()
                guard let self = self else { return }
                self.previewFileTool.openXLSXRemoteFile(urlString: resultUrl, fileName: model.name)
            }
            
            let key = self.getUrlKey(model)
            UserDefaults.standard.set(true, forKey: key)
            UserDefaults.standard.synchronize()
            self.updateModel(model)
        }
    }
    
    func getUrlKey(_ model: YHPrincipleEnvelopeModel) -> String {
        let key = "CustomerServiceFile_" + model.url
        return key
    }
    
    func updateModel(_ model: YHPrincipleEnvelopeModel) {
        
        itemModel = model
        whiteView.isHidden = !model.isUnfold
        let imgName = model.isUnfold ? "custom_service_expand_1" : "custom_service_expand_0"
        expandBtn.setImage(UIImage(named: imgName), for: .normal)
        nameLabel.text = model.name
        
        let key = self.getUrlKey(model)
        if let isWatch = UserDefaults.standard.value(forKey: key) as? Bool, isWatch == true {
            envelopImgView.image = UIImage(named: "principle_envelope_see")
            
        } else {
            envelopImgView.image = UIImage(named: "principle_envelope_unsee")
        }
        
        if !model.isUnfold {
            whiteView.removeSubviews()
            whiteView.snp.remakeConstraints { make in
                make.top.equalTo(titleLabel.snp.bottom).offset(0.0)
                make.height.equalTo(0.0)
                make.left.equalTo(20)
                make.right.equalTo(-20)
                make.bottom.equalTo(-32)
            }
            self.setNeedsLayout()
            self.layoutIfNeeded()
            return
        }
        
        whiteView.removeSubviews()
        whiteView.addSubview(self.envelopImgView)
        whiteView.snp.remakeConstraints { make in
            make.top.equalTo(titleLabel.snp.bottom).offset(16.0)
            make.left.equalTo(20)
            make.right.equalTo(-20)
            make.bottom.equalTo(-32)
        }
        envelopImgView.snp.remakeConstraints { make in
            make.top.equalTo(16)
            make.centerX.equalToSuperview()
            make.width.equalTo(194)
            make.height.equalTo(179)
            make.bottom.equalTo(-17)
        }
        self.setNeedsLayout()
        self.layoutIfNeeded()
    }
    
    func setupUI() {
        
        self.selectionStyle = .none
        self.contentView.backgroundColor = .clear
        self.backgroundColor = .clear
        contentView.addSubview(iconImgView)
        contentView.addSubview(titleLabel)
        contentView.addSubview(expandBtn)
        contentView.addSubview(whiteView)
        whiteView.addSubview(self.envelopImgView)

        iconImgView.snp.makeConstraints { make in
            make.left.equalTo(20)
            make.width.height.equalTo(15.0)
            make.centerY.equalTo(titleLabel)
        }
        
        titleLabel.snp.makeConstraints { make in
            make.left.equalTo(iconImgView.snp.right).offset(6)
            make.right.equalTo(expandBtn.snp.left)
            make.top.equalTo(0)
            make.height.equalTo(20)
        }
        
        expandBtn.snp.makeConstraints { make in
            make.width.height.equalTo(20.0)
            make.centerY.equalTo(titleLabel)
            make.right.equalTo(-20.0)
        }
        
        whiteView.snp.makeConstraints { make in
            make.top.equalTo(titleLabel.snp.bottom).offset(0.0)
            make.height.equalTo(0.0)
            make.left.equalTo(20)
            make.right.equalTo(-20)
            make.bottom.equalTo(-32)
        }
        envelopImgView.snp.makeConstraints { make in
            make.top.equalTo(16)
            make.centerX.equalToSuperview()
            make.width.equalTo(194)
            make.height.equalTo(179)
        }
    }
}
