//
//  YHCertificateInfoCell.swift
//  galaxy
//
//  Created by edy on 2024/3/13.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit
import AttributedString

class YHCertificateInfoCell: UITableViewCell {

    static let cellReuseIdentifier = "YHCertificateInfoCell"
    var model: YHSupplementInfo = YHSupplementInfo()
    var status: YHCertificateUploadStatus = .preUpload
    var templateBlock: (() -> Void)?
    var whiteView: UIView!
    var nameTextView: UITextView!
    var infoLabel: UILabel!
    var statusLabel: UILabel!
    var rightArrowImgV: UIImageView!
    lazy var templateButton: UIButton = {
        let btn = UIButton()
        btn.frame = CGRect(x: 0, y: 0, width: 48, height: 16)
        btn.setTitle("资料模板", for: .normal)
        btn.titleLabel?.font = .PFSC_M(ofSize: 10)
        btn.setTitleColor(.brandMainColor, for: .normal)
        btn.layer.cornerRadius = kCornerRadius3
        btn.clipsToBounds = true
        btn.backgroundColor = .brandMainColor8
        btn.addTarget(self, action: #selector(didTemplateBtnClicked), for: .touchUpInside)
        btn.YH_clickEdgeInsets = UIEdgeInsets(top: 20, left: 20, bottom: 20, right: 20)
        return btn
    }()

    required init?(coder: NSCoder) {
        super.init(coder: coder)
    }
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        setupUI()
    }
    
    func updateModel(_ model: YHSupplementInfo) {
        
        self.model = model
        if let status = YHCertificateUploadStatus(rawValue: model.checkStatus) {
            
            var titleColor: UIColor = .labelTextColor2
            var title = ""
            switch status {
            case .preUpload:
                titleColor = .brandMainColor
                title = "待上传".local
            case .rejected:
                titleColor = .failColor
                title = "已驳回".local
            case .review:
                titleColor = .warnColor
                title = "审核中".local
            case .finish:
                titleColor = UIColor(hex: 0x3CC694)
                title = "已完成".local
            case .all:
                titleColor = .labelTextColor2
            }
            statusLabel.text = title
            statusLabel.textColor = titleColor
        }
        
        var infoStr = ""
        var typeStr = "无"
        var ownerStr = "无"
        if !model.typeName.isEmpty {
            typeStr = model.typeName
        }
        if !model.nameExtra.applicant.isEmpty {
            ownerStr = model.nameExtra.applicant
        }
        
        infoStr = "类型: \(typeStr)\n所属人: \(ownerStr)"
        let attributes: [NSAttributedString.Key: Any] = [
            .font: UIFont.PFSC_R(ofSize: 12),
            .foregroundColor: UIColor.labelTextColor2
        ]
        let attrString = NSMutableAttributedString(string: infoStr, attributes: attributes)
        let paragraphStyle = NSMutableParagraphStyle()
        paragraphStyle.lineSpacing = 4.0 // 设置行间距为 8 点
        attrString.addAttribute(.paragraphStyle, value: paragraphStyle, range: NSRange(location: 0, length: infoStr.count))
        
        infoLabel.attributedText = attrString

        var result: ASAttributedString = .init(string: "", .font(UIFont.PFSC_M(ofSize: 16)), .foreground(UIColor.mainTextColor))
        let mustTag: ASAttributedString = .init("*", .font(UIFont.PFSC_M(ofSize: 16)), .foreground(UIColor.failColor))
        var name = model.name
        if !model.nameExtra.organization.isEmpty {
            name += "(\(model.nameExtra.organization))"
        }
        let cerName: ASAttributedString = .init(string: name, .font(UIFont.PFSC_M(ofSize: 16)), .foreground(UIColor.mainTextColor))
        var template: ASAttributedString = " \(.view(templateButton, .original(.center)))"
        
        // 是否展示右边的模板按钮
        var showTemplateBtn = true
        if model.isNeedDealt {
            templateButton.setTitle("银河代办", for: .normal)
            template = " \(.view(templateButton, .original(.center)))"
        } else {
            templateButton.setTitle("资料模板", for: .normal)
            template = " \(.view(templateButton, .original(.center)))"

            if model.templateInfo.materialPath.isEmpty && model.templateInfo.guideFilePath.isEmpty && model.templateInfo.description.isEmpty {
                
                showTemplateBtn = false
            }
        }
        
        if model.isNeedSupplemnt() {
            result += mustTag
        }
        result += cerName
        if showTemplateBtn {
            result += template
        }
        nameTextView.attributed.text = result
    }
    
    @objc func didTemplateBtnClicked() {
        if self.model.isNeedDealt { // 银河代办
            return
        }
        if let templateBlock = templateBlock {
            templateBlock()
        }
    }
    
    func setupUI() {
        
        self.selectionStyle = .none
        contentView.backgroundColor = UIColor(hexString: "#F8F8F8")
        
        whiteView = UIView()
        whiteView.layer.cornerRadius = kCornerRadius6
        whiteView.clipsToBounds = true
        whiteView.backgroundColor = .white
        contentView.addSubview(whiteView)
        
        nameTextView = UITextView()
        nameTextView.textColor = UIColor.mainTextColor
        nameTextView.font = UIFont.PFSC_M(ofSize: 16)
        nameTextView.isScrollEnabled = false
        nameTextView.isEditable = false
        nameTextView.isSelectable = false
        whiteView.addSubview(nameTextView)
        
        nameTextView.attributed.text = """
        \("资料名称", .foreground(UIColor.mainTextColor), .font(UIFont.PFSC_M(ofSize: 16))) \(.view(templateButton, .original(.center))))
        """
        
        infoLabel = UILabel()
        infoLabel.textColor = UIColor.labelTextColor2
        infoLabel.textAlignment = .left
        infoLabel.font = UIFont.PFSC_R(ofSize: 11)
        infoLabel.numberOfLines = 0
        infoLabel.setContentHuggingPriority(.required, for: .vertical)
        infoLabel.setContentCompressionResistancePriority(.required, for: .vertical)
        whiteView.addSubview(infoLabel)
        
        statusLabel = UILabel()
        statusLabel.textColor = UIColor.labelTextColor2
        statusLabel.textAlignment = .left
        statusLabel.font = UIFont.PFSC_R(ofSize: 14)
        whiteView.addSubview(statusLabel)
        
        rightArrowImgV = UIImageView(image: UIImage(named: "service_center_right_next_icon"))
        whiteView.addSubview(rightArrowImgV)
        
        whiteView.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(20)
            make.right.equalToSuperview().offset(-20)
            make.top.bottom.equalToSuperview()
        }
        
        nameTextView.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(10)
            make.top.equalToSuperview().offset(4)
            make.right.equalTo(statusLabel.snp.left).offset(-21)
        }
        
        infoLabel.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(16)
            make.right.equalTo(nameTextView)
            make.top.equalTo(nameTextView.snp.bottom)
            make.bottom.equalToSuperview().offset(-12)
        }
        
        statusLabel.snp.makeConstraints { make in
            make.right.equalTo(rightArrowImgV.snp.left).offset(-6)
            make.size.equalTo(CGSize(width: 42, height: 20))
            make.centerY.equalToSuperview()
        }
        
        rightArrowImgV.snp.makeConstraints { make in
            make.right.equalToSuperview().offset(-16)
            make.size.equalTo(CGSize(width: 16, height: 16))
            make.centerY.equalToSuperview()
        }
    }
    
    override func hitTest(_ point: CGPoint, with event: UIEvent?) -> UIView? {
        let hitView = super.hitTest(point, with: event)
        if hitView == nameTextView {
            return self
        }
        return hitView
    }
}
