//
//  YHHKRecordsRemarkCell.swift
//  galaxy
//
//  Created by alexzzw on 2024/9/19.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

class YHHKRecordsRemarkCell: YHResignDocumentCell {

    static let cellReuseIdentifier = "YHHKRecordsRemarkCell"
    private let maxCount = 200
    
    var textViewDidChanged: ((UITextView) -> Void)?
    var textDidExceedsMaxLimit: ((Int) -> Void)?
    
    private lazy var infoTitleLabel: UILabel = {
        let label = UILabel()
        label.font = .PFSC_R(ofSize: 14)
        label.textColor = .mainTextColor50
        return label
    }()
    
    private lazy var remarkTextViewContainer: YHBaseDynamicCornerRadiusView = {
        let view = YHBaseDynamicCornerRadiusView(cornerRadius: 4, corner: .allCorners)
        view.backgroundColor = .contentBkgColor
        return view
    }()
    
    private lazy var remarkTextView: UITextView = {
        let view = UITextView()
        view.backgroundColor = .clear
        view.font = .PFSC_R(ofSize: 14)
        view.textColor = .mainTextColor
        view.tintColor = .mainTextColor
        view.returnKeyType = .default
        view.enablesReturnKeyAutomatically = true
        view.delegate = self
        view.textContainerInset = .zero
        view.textContainer.lineFragmentPadding = 0
        return view
    }()
    
    private lazy var editTipsLabel: UILabel = {
        let label = UILabel()
        label.isHidden = true
        label.text = "请输入备注内容"
        label.font = UIFont.systemFont(ofSize: 14)
        label.textColor = .mainTextColor30
        return label
    }()

    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        setupUI()
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    func setupCellInfo(title: String? = "备注：", titleColor: UIColor = .mainTextColor50, detail: String?, isEditable: Bool = false, cellType: YHResignRoundCellType = .mid) {
        updateCellCorner(cellType)
        infoTitleLabel.text = title
        infoTitleLabel.textColor = titleColor
        remarkTextView.isEditable = isEditable
        remarkTextView.text = detail
        editTipsLabel.isHidden = !isEditable || (detail?.count ?? 0) > 0
    }
}

extension YHHKRecordsRemarkCell {
    
    private func setupUI() {
        subContainerView.addSubview(infoTitleLabel)
        subContainerView.addSubview(remarkTextViewContainer)
        remarkTextViewContainer.addSubview(remarkTextView)
        subContainerView.addSubview(editTipsLabel)
        
        infoTitleLabel.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(18)
            make.top.equalToSuperview().offset(16)
        }
        
        remarkTextViewContainer.snp.makeConstraints { make in
            make.left.equalTo(infoTitleLabel)
            make.right.equalToSuperview().offset(-18)
            make.height.equalTo(86).priority(.high)
            make.top.equalTo(infoTitleLabel.snp.bottom).offset(8)
            make.bottom.equalToSuperview().offset(-16)
        }
        
        remarkTextView.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(12)
            make.right.equalToSuperview().offset(-12)
            make.top.equalToSuperview().offset(10)
            make.bottom.equalToSuperview().offset(-10).priority(.high)
        }

        editTipsLabel.snp.makeConstraints { make in
            make.left.equalTo(remarkTextViewContainer.snp.left).offset(12)
            make.top.equalTo(remarkTextViewContainer.snp.top).offset(10)
        }
    }
}

extension YHHKRecordsRemarkCell: UITextViewDelegate {
    
    func textViewDidChange(_ textView: UITextView) {
        editTipsLabel.isHidden = textView.text.count > 0
        textViewDidChanged?(textView)
    }
    
    func textView(_ textView: UITextView, shouldChangeTextIn range: NSRange, replacementText text: String) -> Bool {
        if textView == remarkTextView {
            /// 正在输入拼音
            if let textRange = textView.markedTextRange, range.length == 0 {
                /// 刚输入拼音的时候总字数
                let start = textView.offset(from: textView.beginningOfDocument, to: textRange.start)
                return start <= maxCount
            }
            let count = (textView.text?.utf16.count ?? 0) - range.length + text.utf16.count
            if count > maxCount && text == "" {
                return true
            }
            guard count > maxCount else {
                return true
            }
            textDidExceedsMaxLimit?(maxCount)
            return false
        }
        return true
    }
}
