//
//  YHResignDocumentStatusCell.swift
//  galaxy
//
//  Created by alexzzw on 2024/9/13.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

class YHResignDocumentStatusCell: YHResignDocumentCell {
    enum ButtonsLayoutStyle: Int {
        case one = 0, two, three, four
    }
    
    struct ButtonsConfig {
        var style: ButtonsLayoutStyle = .one
        var names: [String] = []
        var actions: [(() -> Void)] = []
    }

    static let cellReuseIdentifier = "YHResignDocumentStatusCell"
    private var leftButtonClickEvent: (() -> Void)?
    private var midButtonClickEvent: (() -> Void)?
    private var rightButtonClickEvent: (() -> Void)?
    private var farRightButtonClickEvent: (() -> Void)?
    
    private lazy var infoTitleLabel: UILabel = {
        let label = UILabel()
        label.font = .PFSC_M(ofSize: 17)
        label.textColor = UIColor(hexString: "#222222")
        label.text = "文书稿件"
        return label
    }()
    
    private lazy var infoStatusLabel: YHPaddedLabel = {
        let label = YHPaddedLabel(padding: UIEdgeInsets(top: 0, left: 4, bottom: 0, right: 4))
        label.font = .PFSC_M(ofSize: 10)
        label.layer.cornerRadius = 2
        label.layer.masksToBounds = true
        return label
    }()
    
    private lazy var topLineView: UIView = {
        let view = UIView()
        view.backgroundColor = .separatorColor
        return view
    }()
    
    private lazy var fileCoverView: UIView = {
        let view = UIView()
        return view
    }()
    
    private lazy var fileIconView: UIImageView = {
        let view = UIImageView()
        view.image = UIImage(named: "my_cer_type_word")
        return view
    }()
    
    private lazy var fileNameLabel: UILabel = {
        let label = UILabel()
        label.textColor = UIColor(hexString: "#222222")
        label.font = .PFSC_M(ofSize: 14)
        label.numberOfLines = 0
        return label
    }()
    
    private lazy var bottomLineView: UIView = {
        let view = UIView()
        view.backgroundColor = .separatorColor
        return view
    }()
    
    private lazy var buttonsCoverView: UIView = {
        let view = UIView()
        return view
    }()
    
    private lazy var leftButton: UIButton = {
        let button = UIButton(type: .custom)
        button.isHidden = true
        button.setTitleColor(UIColor.brandMainColor, for: .normal)
        button.titleLabel?.font = .PFSC_R(ofSize: 14)
        button.addTarget(self, action: #selector(leftButtonDidClick(_:)), for: .touchUpInside)
        return button
    }()
    
    private lazy var midButton: UIButton = {
        let button = UIButton(type: .custom)
        button.isHidden = true
        button.setTitleColor(UIColor.brandMainColor, for: .normal)
        button.titleLabel?.font = .PFSC_R(ofSize: 14)
        button.addTarget(self, action: #selector(midButtonDidClick(_:)), for: .touchUpInside)
        return button
    }()
    
    private lazy var rightButton: UIButton = {
        let button = UIButton(type: .custom)
        button.isHidden = true
        button.setTitleColor(UIColor.brandMainColor, for: .normal)
        button.titleLabel?.font = .PFSC_R(ofSize: 14)
        button.addTarget(self, action: #selector(rightButtonDidClick(_:)), for: .touchUpInside)
        return button
    }()
    
    private lazy var farRightButton: UIButton = {
        let button = UIButton(type: .custom)
        button.isHidden = true
        button.setTitleColor(UIColor.brandMainColor, for: .normal)
        button.titleLabel?.font = .PFSC_R(ofSize: 14)
        button.addTarget(self, action: #selector(farRightButtonDidClick(_:)), for: .touchUpInside)
        return button
    }()
    
    private lazy var leftLineView: UIView = {
        let view = UIView()
        view.isHidden = true
        view.backgroundColor = .separatorColor
        return view
    }()
    
    private lazy var rightLineView: UIView = {
        let view = UIView()
        view.isHidden = true
        view.backgroundColor = .separatorColor
        return view
    }()
    
    private lazy var farRightLineView: UIView = {
        let view = UIView()
        view.isHidden = true
        view.backgroundColor = .separatorColor
        return view
    }()

    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        setupUI()
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    func setupCellInfo(title: String, fileName: String, status: YHResignDocumentStatus, buttonsConfig: ButtonsConfig) {
        infoTitleLabel.text = title
        infoStatusLabel.text = status.title()
        infoStatusLabel.textColor = status.textColor()
        infoStatusLabel.backgroundColor = status.statusLabelBgColor()
        fileNameLabel.text = fileName
        // updateFileIcon(fileName)
        updateButtons(buttonsConfig)
    }
}

extension YHResignDocumentStatusCell {
    @objc private func leftButtonDidClick(_ sender: UIButton) {
        leftButtonClickEvent?()
    }
    
    @objc private func midButtonDidClick(_ sender: UIButton) {
        midButtonClickEvent?()
    }
    
    @objc private func rightButtonDidClick(_ sender: UIButton) {
        rightButtonClickEvent?()
    }
    
    @objc private func farRightButtonDidClick(_ sender: UIButton) {
        farRightButtonClickEvent?()
    }
}

extension YHResignDocumentStatusCell {
    
    private func updateFileIcon(_ fileName: String) {
        let suffix = fileName.pathExtension.lowercased()
        var iconImgName = ""
        if suffix == "jpeg" {
            iconImgName = "my_cer_type_jpg"
        } else if suffix == "jpg" {
            iconImgName = "my_cer_type_jpg"
        } else if suffix == "png" {
            iconImgName = "my_cer_type_png"
        } else if suffix == "pdf" {
            iconImgName = "my_cer_type_pdf"
        } else if suffix == "doc" || suffix == "docx" {
            iconImgName = "my_cer_type_word"
        } else if suffix == "ppt" || suffix == "pptx" {
            iconImgName = "my_cer_type_ppt"
        } else if suffix == "bmp" {
            iconImgName = "my_cer_type_bmp"
        } else if suffix == "tiff" {
            iconImgName = "my_cer_type_tiff"
        } else if suffix == "xls" || suffix == "xlsx" {
            iconImgName = "my_cer_type_excel"
        }
        fileIconView.image = UIImage(named: iconImgName)
    }
    
    private func updateButtons(_ buttonsConfig: ButtonsConfig) {
        leftButton.removeTarget(self, action: #selector(leftButtonDidClick(_:)), for: .touchUpInside)
        midButton.removeTarget(self, action: #selector(midButtonDidClick(_:)), for: .touchUpInside)
        rightButton.removeTarget(self, action: #selector(rightButtonDidClick(_:)), for: .touchUpInside)
        
        switch buttonsConfig.style {
        case .one:
            leftButton.isHidden = false
            midButton.isHidden = true
            rightButton.isHidden = true
            leftLineView.isHidden = true
            rightLineView.isHidden = true
            farRightButton.isHidden = true
            farRightLineView.isHidden = true
            
            if let first = buttonsConfig.names.first {
                leftButton.setTitle(first, for: .normal)
            }
            if let first = buttonsConfig.actions.first {
                leftButtonClickEvent = first
                leftButton.addTarget(self, action: #selector(leftButtonDidClick(_:)), for: .touchUpInside)
            }
            leftButton.snp.remakeConstraints { make in
                make.left.top.bottom.right.equalToSuperview()
            }
            midButton.snp.remakeConstraints { make in
                make.left.top.equalToSuperview()
                make.width.height.equalTo(0)
            }
            rightButton.snp.remakeConstraints { make in
                make.left.top.equalToSuperview()
                make.width.height.equalTo(0)
            }
            farRightButton.snp.remakeConstraints { make in
                make.left.top.equalToSuperview()
                make.width.height.equalTo(0)
            }
        case .two:
            leftButton.isHidden = false
            midButton.isHidden = false
            rightButton.isHidden = true
            leftLineView.isHidden = false
            rightLineView.isHidden = true
            farRightButton.isHidden = true
            farRightLineView.isHidden = true
            
            if buttonsConfig.names.count > 1 {
                leftButton.setTitle(buttonsConfig.names[0], for: .normal)
                midButton.setTitle(buttonsConfig.names[1], for: .normal)
            }
            if buttonsConfig.actions.count > 1 {
                leftButtonClickEvent = buttonsConfig.actions[0]
                leftButton.addTarget(self, action: #selector(leftButtonDidClick(_:)), for: .touchUpInside)
                midButtonClickEvent = buttonsConfig.actions[1]
                midButton.addTarget(self, action: #selector(midButtonDidClick(_:)), for: .touchUpInside)
            }
            
            leftButton.snp.remakeConstraints { make in
                make.left.top.bottom.equalToSuperview()
            }
            leftLineView.snp.remakeConstraints { make in
                make.left.equalTo(leftButton.snp.right)
                make.height.equalTo(16)
                make.width.equalTo(0.5).priority(.high)
                make.centerY.equalToSuperview()
            }
            midButton.snp.remakeConstraints { make in
                make.left.equalTo(leftLineView.snp.right)
                make.top.bottom.right.equalToSuperview()
                make.width.equalTo(leftButton)
            }
            rightButton.snp.remakeConstraints { make in
                make.left.top.equalToSuperview()
                make.width.height.equalTo(0)
            }
            farRightButton.snp.remakeConstraints { make in
                make.left.top.equalToSuperview()
                make.width.height.equalTo(0)
            }
            
        case .three:
            leftButton.isHidden = false
            midButton.isHidden = false
            rightButton.isHidden = false
            leftLineView.isHidden = false
            rightLineView.isHidden = false
            farRightButton.isHidden = true
            farRightLineView.isHidden = true
            
            if buttonsConfig.names.count > 2 {
                leftButton.setTitle(buttonsConfig.names[0], for: .normal)
                midButton.setTitle(buttonsConfig.names[1], for: .normal)
                rightButton.setTitle(buttonsConfig.names[2], for: .normal)
            }
            if buttonsConfig.actions.count > 2 {
                leftButtonClickEvent = buttonsConfig.actions[0]
                leftButton.addTarget(self, action: #selector(leftButtonDidClick(_:)), for: .touchUpInside)
                midButtonClickEvent = buttonsConfig.actions[1]
                midButton.addTarget(self, action: #selector(midButtonDidClick(_:)), for: .touchUpInside)
                rightButtonClickEvent = buttonsConfig.actions[2]
                rightButton.addTarget(self, action: #selector(rightButtonDidClick(_:)), for: .touchUpInside)
            }
            
            leftButton.snp.remakeConstraints { make in
                make.left.top.bottom.equalToSuperview()
            }
            
            leftLineView.snp.remakeConstraints { make in
                make.left.equalTo(leftButton.snp.right)
                make.height.equalTo(16)
                make.width.equalTo(0.5).priority(.high)
                make.centerY.equalToSuperview()
            }
            
            midButton.snp.remakeConstraints { make in
                make.left.equalTo(leftLineView.snp.right)
                make.top.bottom.equalToSuperview()
                make.width.equalTo(leftButton)
            }
            
            rightLineView.snp.remakeConstraints { make in
                make.left.equalTo(midButton.snp.right)
                make.height.equalTo(16)
                make.width.equalTo(0.5).priority(.high)
                make.centerY.equalToSuperview()
            }
            
            rightButton.snp.remakeConstraints { make in
                make.left.equalTo(rightLineView.snp.right)
                make.top.right.bottom.equalToSuperview()
                make.width.equalTo(leftButton)
            }
            
            farRightButton.snp.remakeConstraints { make in
                make.left.top.equalToSuperview()
                make.width.height.equalTo(0)
            }
        case .four:
            leftButton.isHidden = false
            midButton.isHidden = false
            rightButton.isHidden = false
            leftLineView.isHidden = false
            rightLineView.isHidden = false
            farRightButton.isHidden = false
            farRightLineView.isHidden = false
            
            if buttonsConfig.names.count > 3 {
                leftButton.setTitle(buttonsConfig.names[0], for: .normal)
                midButton.setTitle(buttonsConfig.names[1], for: .normal)
                rightButton.setTitle(buttonsConfig.names[2], for: .normal)
                farRightButton.setTitle(buttonsConfig.names[3], for: .normal)
            }
            if buttonsConfig.actions.count > 3 {
                leftButtonClickEvent = buttonsConfig.actions[0]
                leftButton.addTarget(self, action: #selector(leftButtonDidClick(_:)), for: .touchUpInside)
                midButtonClickEvent = buttonsConfig.actions[1]
                midButton.addTarget(self, action: #selector(midButtonDidClick(_:)), for: .touchUpInside)
                rightButtonClickEvent = buttonsConfig.actions[2]
                rightButton.addTarget(self, action: #selector(rightButtonDidClick(_:)), for: .touchUpInside)
                farRightButtonClickEvent = buttonsConfig.actions[3]
                farRightButton.addTarget(self, action: #selector(farRightButtonDidClick(_:)), for: .touchUpInside)
            }
            
            leftButton.snp.remakeConstraints { make in
                make.left.top.bottom.equalToSuperview()
            }
            
            leftLineView.snp.remakeConstraints { make in
                make.left.equalTo(leftButton.snp.right)
                make.height.equalTo(16)
                make.width.equalTo(0.5).priority(.high)
                make.centerY.equalToSuperview()
            }
            
            midButton.snp.remakeConstraints { make in
                make.left.equalTo(leftLineView.snp.right)
                make.top.bottom.equalToSuperview()
                make.width.equalTo(leftButton)
            }
            
            rightLineView.snp.remakeConstraints { make in
                make.left.equalTo(midButton.snp.right)
                make.height.equalTo(16)
                make.width.equalTo(0.5).priority(.high)
                make.centerY.equalToSuperview()
            }
            
            rightButton.snp.remakeConstraints { make in
                make.left.equalTo(rightLineView.snp.right)
                make.top.bottom.equalToSuperview()
                make.width.equalTo(leftButton)
            }
            
            farRightLineView.snp.remakeConstraints { make in
                make.left.equalTo(rightButton.snp.right)
                make.height.equalTo(16)
                make.width.equalTo(0.5).priority(.high)
                make.centerY.equalToSuperview()
            }
            
            farRightButton.snp.remakeConstraints { make in
                make.left.equalTo(farRightLineView.snp.right)
                make.top.right.bottom.equalToSuperview()
                make.width.equalTo(leftButton)
            }
        }
    }
    
    private func setupUI() {
        updateCellCorner(.single)
        subContainerView.addSubview(infoTitleLabel)
        subContainerView.addSubview(infoStatusLabel)
        subContainerView.addSubview(topLineView)
        subContainerView.addSubview(fileCoverView)
        fileCoverView.addSubview(fileIconView)
        fileCoverView.addSubview(fileNameLabel)
        subContainerView.addSubview(bottomLineView)
        subContainerView.addSubview(buttonsCoverView)
        buttonsCoverView.addSubviews([leftButton, leftLineView, midButton, rightLineView, rightButton, farRightLineView, farRightButton])
        
        infoTitleLabel.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(18)
            make.top.equalToSuperview().offset(16)
        }
        
        infoStatusLabel.snp.makeConstraints { make in
            make.left.equalTo(infoTitleLabel.snp.right).offset(4)
            make.height.equalTo(18)
            make.centerY.equalTo(infoTitleLabel)
        }
        
        topLineView.snp.makeConstraints { make in
            make.left.equalTo(infoTitleLabel.snp.left)
            make.top.equalTo(infoTitleLabel.snp.bottom).offset(12)
            make.right.equalToSuperview().offset(-18)
            make.height.equalTo(0.5)
        }
        
        fileCoverView.snp.makeConstraints { make in
            make.top.equalTo(topLineView.snp.bottom)
            make.left.equalTo(topLineView.snp.left)
            make.right.equalTo(topLineView.snp.right)
            make.height.greaterThanOrEqualTo(72)
        }
        
        fileIconView.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(6)
            make.centerY.equalToSuperview()
            make.width.height.equalTo(31).priority(.high)
        }
        
        fileNameLabel.snp.makeConstraints { make in
            make.top.greaterThanOrEqualToSuperview().offset(16)
            make.left.equalTo(fileIconView.snp.right).offset(12)
            make.right.lessThanOrEqualToSuperview()
            make.centerY.equalToSuperview()
            make.bottom.lessThanOrEqualToSuperview().offset(-16)
        }
        
        bottomLineView.snp.makeConstraints { make in
            make.left.equalToSuperview()
            make.top.equalTo(fileCoverView.snp.bottom)
            make.right.equalToSuperview()
            make.height.equalTo(0.5)
        }
        
        buttonsCoverView.snp.makeConstraints { make in
            make.left.right.equalToSuperview()
            make.top.equalTo(bottomLineView.snp.bottom)
            make.height.equalTo(52)
            make.bottom.equalToSuperview()
        }
        
        leftButton.snp.makeConstraints { make in
            make.left.top.bottom.equalToSuperview()
        }
        
        leftLineView.snp.makeConstraints { make in
            make.left.equalTo(leftButton.snp.right)
            make.height.equalTo(16)
            make.width.equalTo(0.5).priority(.high)
            make.centerY.equalToSuperview()
        }
        
        midButton.snp.makeConstraints { make in
            make.left.equalTo(leftLineView.snp.right)
            make.top.bottom.equalToSuperview()
            make.width.equalTo(leftButton)
        }
        
        rightLineView.snp.makeConstraints { make in
            make.left.equalTo(midButton.snp.right)
            make.height.equalTo(16)
            make.width.equalTo(0.5).priority(.high)
            make.centerY.equalToSuperview()
        }
        
        rightButton.snp.makeConstraints { make in
            make.left.equalTo(rightLineView.snp.right)
            make.top.bottom.equalToSuperview()
            make.width.equalTo(leftButton)
        }
        
        farRightLineView.snp.makeConstraints { make in
            make.left.equalTo(rightButton.snp.right)
            make.height.equalTo(16)
            make.width.equalTo(0.5).priority(.high)
            make.centerY.equalToSuperview()
        }
        
        farRightButton.snp.makeConstraints { make in
            make.left.equalTo(farRightLineView.snp.right)
            make.top.right.bottom.equalToSuperview()
            make.width.equalTo(leftButton)
        }
    }
}

private extension YHResignDocumentStatus {
    
    func statusLabelBgColor() -> UIColor? {
        switch self {
        case .pendingConfirmation:
            UIColor.tipsColor8
        case .finalizing:
            UIColor(hexString: "#FFF5EB")
        case .underReview:
            UIColor(hexString: "#FFF5EB")
        case .awaitingSignature:
            UIColor.tipsColor8
        case .completed:
            UIColor(hexString: "#F0FBF7")
        }
    }
    
}
