//
//  YHResignScheduleMultipleInfoCell.swift
//  galaxy
//
//  Created by edy on 2024/10/10.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

class YHResignScheduleMultipleInfoCell: UITableViewCell {

    static let cellReuseIdentifier = "YHResignScheduleMultipleInfoCell"
    
    var confirmHKStatus: YHResignConfirmHKStatus = .waitConfirmHK
    var clickConfirmBtnBlock: ((YHResignConfirmHKStatus) -> Void)?
    var applyPaymentBlock: (() -> Void)?
    
    lazy var whiteCornerView: UIView = {
        let view = UIView()
        view.backgroundColor = .white
        view.layer.cornerRadius = kCornerRadius6
        view.clipsToBounds = true
        return view
    }()

    lazy var lineView: UIView = {
        let view = UIView()
        view.backgroundColor = .separatorColor
        return view
    }()
    
    lazy var markView: UIView = {
        let view = UIView()
        view.backgroundColor = .brandMainColor
        return view
    }()
    
    lazy var titleLabel: UILabel = {
        let label = UILabel()
        label.font = UIFont.PFSC_M(ofSize: 15)
        label.textColor = .mainTextColor
        label.text = "第X批"
        return label
    }()
    
    lazy var haveConfirmInHKLabel: UILabel = {
        let label = UILabel()
        label.text = "已确认在港"
        label.font = UIFont.PFSC_M(ofSize: 13)
        label.textColor = .init(hex: 0x3CC694)
        return label
    }()
    
    lazy var confirmInHKBtn: UIButton = {
        let btn = UIButton()
        btn.layer.cornerRadius = 2.0
        btn.backgroundColor = .brandMainColor
        btn.setTitleColor(.white, for: .normal)
        btn.setTitle("确认在港", for: .normal)
        btn.titleLabel?.font = .PFSC_M(ofSize: 11)
        btn.addTarget(self, action: #selector(didConfirmInHKBtnClicked), for: .touchUpInside)
        return btn
    }()
    
    lazy var applyFeeBtn: UIButton = {
        let btn = UIButton()
        btn.layer.cornerRadius = 2.0
        btn.backgroundColor = .brandMainColor
        btn.setTitleColor(.white, for: .normal)
        btn.setTitle("申请缴费", for: .normal)
        btn.titleLabel?.font = .PFSC_M(ofSize: 11)
        btn.addTarget(self, action: #selector(didApplyFeeBtnClicked), for: .touchUpInside)
        return btn
    }()
    
    lazy var applicantNamesLabel: UILabel = {
        let label = UILabel()
        label.font = UIFont.PFSC_M(ofSize: 14)
        label.textColor = .mainTextColor
        label.text = "申请人："
        return label
    }()
    
    lazy var submitTimeLabel: UILabel = {
        let label = UILabel()
        label.font = UIFont.PFSC_R(ofSize: 12)
        label.textColor = .mainTextColor(alpha: 0.5)
        label.text = "预约递交时间: 0000-00-00 00:00"
        return label
    }()
    
    lazy var confirmHKTimeLabel: UILabel = {
        let label = UILabel()
        label.font = UIFont.PFSC_R(ofSize: 12)
        label.textColor = .mainTextColor(alpha: 0.5)
        label.text = "确认在港时间: 0000-00-00 00:00"
        return label
    }()
    
    lazy var scheduleView: YHResignAppointedScheduleLineView = {
        let view = YHResignAppointedScheduleLineView(frame: .zero)
        return view
    }()
    
    required init?(coder: NSCoder) {
        super.init(coder: coder)
    }
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        createUI()
    }
    
    @objc func didConfirmInHKBtnClicked() {
        clickConfirmBtnBlock?(self.confirmHKStatus)
    }
    
    @objc func didApplyFeeBtnClicked() {
        applyPaymentBlock?()
    }
    
    func updateModel(_ model: YHResignAppointGroup) {
        
        var lineMargin = 0.0
        if model.batchIndex > 1 {
            lineMargin = 18.0
        }
        lineView.snp.updateConstraints { make in
            make.left.equalToSuperview().offset(lineMargin)
            make.right.equalToSuperview().offset(-lineMargin)
        }
        
        titleLabel.text = "第\(String.convertNumberToChineseText(model.batchIndex))批"
        var names = "申请人："
        for (index, user) in model.user_list.enumerated() {
            names += user.name
            if index != model.user_list.count-1 {
                names += "、"
            }
        }
        applicantNamesLabel.text = names
        submitTimeLabel.text = "预约递交时间: " + model.getSubmitHKTime()
        confirmHKTimeLabel.text = "确认在港时间: " + model.getConfirmHKTime()
        
        self.confirmHKStatus = model.getConfirmInHKStatus()
        
        haveConfirmInHKLabel.isHidden = (self.confirmHKStatus != .haveConfirmHK)
        confirmInHKBtn.isHidden = (self.confirmHKStatus == .haveConfirmHK)
        
        if self.confirmHKStatus == .waitConfirmHK {
            confirmInHKBtn.setTitle("待确认在港", for: .normal)
            confirmInHKBtn.setTitleColor(.mainTextColor, for: .normal)
            confirmInHKBtn.backgroundColor = .init(hex: 0xF4F6FA)
            confirmInHKBtn.snp.updateConstraints { make in
                make.width.equalTo(71)
            }
            
        } else if self.confirmHKStatus == .canConfirmHK {
            confirmInHKBtn.setTitle("确认在港", for: .normal)
            confirmInHKBtn.setTitleColor(.white, for: .normal)
            confirmInHKBtn.backgroundColor = .brandMainColor
            confirmInHKBtn.snp.updateConstraints { make in
                make.width.equalTo(68)
            }
        }
        
        confirmHKTimeLabel.snp.removeConstraints()
        confirmHKTimeLabel.isHidden = true
        applyFeeBtn.isHidden = !model.isNeedPayment()
        haveConfirmInHKLabel.snp.remakeConstraints { make in
            if model.isNeedPayment() {
                make.right.equalTo(applyFeeBtn.snp.left).offset(-8)
                
            } else {
                make.right.equalTo(-18)
            }
            make.centerY.equalTo(titleLabel)
            make.height.equalTo(18)
        }
        
        if self.confirmHKStatus == .haveConfirmHK { // 已确认在港 需要显示确认在港时间
            confirmHKTimeLabel.isHidden = false
            confirmHKTimeLabel.snp.remakeConstraints { make in
                make.left.equalTo(18)
                make.right.equalTo(-18)
                make.top.equalTo(submitTimeLabel.snp.bottom).offset(8)
            }
            scheduleView.snp.remakeConstraints { make in
                make.left.equalTo(18)
                make.right.equalTo(-18)
                make.top.equalTo(confirmHKTimeLabel.snp.bottom).offset(18)
                make.height.equalTo(84.0)
                make.bottom.equalTo(-18)
            }
            
        } else { // 不需要显示确认在港时间
            
            scheduleView.snp.remakeConstraints { make in
                make.left.equalTo(18)
                make.right.equalTo(-18)
                make.top.equalTo(submitTimeLabel.snp.bottom).offset(18)
                make.height.equalTo(84.0)
                make.bottom.equalTo(-18)
            }
        }
        
        scheduleView.updateDates(confirmDate: model.determine_hk_at, submitDate: model.actually_submit_at, watchDate: model.result_file_at)
        
        self.setNeedsLayout()
        self.layoutIfNeeded()
    }
    
    func createUI() {
        
        self.selectionStyle = .none
        self.backgroundColor = .clear
        contentView.backgroundColor = .clear
        contentView.clipsToBounds = true
        contentView.addSubview(whiteCornerView)
        contentView.addSubview(lineView)
        contentView.addSubview(markView)
        contentView.addSubview(titleLabel)
        contentView.addSubview(confirmInHKBtn)
        contentView.addSubview(applyFeeBtn)
        contentView.addSubview(haveConfirmInHKLabel)
        contentView.addSubview(confirmHKTimeLabel)
        contentView.addSubview(applicantNamesLabel)
        contentView.addSubview(submitTimeLabel)
        contentView.addSubview(confirmHKTimeLabel)
        contentView.addSubview(scheduleView)
        
        whiteCornerView.snp.makeConstraints { make in
            make.top.equalTo(-kCornerRadius6)
            make.left.right.bottom.equalToSuperview()
        }

        lineView.snp.makeConstraints { make in
            make.left.right.top.equalToSuperview()
            make.height.equalTo(0.5)
        }
        
        markView.snp.makeConstraints { make in
            make.width.equalTo(3)
            make.height.equalTo(12)
            make.left.equalTo(18)
            make.centerY.equalTo(titleLabel)
        }
        
        titleLabel.snp.makeConstraints { make in
            make.left.equalTo(markView.snp.right).offset(8)
            make.height.equalTo(22)
            make.top.equalTo(18)
        }
        
        confirmInHKBtn.snp.makeConstraints { make in
            make.right.equalTo(-18)
            make.width.equalTo(68)
            make.height.equalTo(28)
            make.top.equalTo(18)
        }
        
        applyFeeBtn.snp.makeConstraints { make in
            make.right.equalTo(-18)
            make.width.equalTo(68)
            make.height.equalTo(28)
            make.centerY.equalTo(haveConfirmInHKLabel)
        }
        
        haveConfirmInHKLabel.snp.makeConstraints { make in
            make.right.equalTo(-18)
            make.centerY.equalTo(titleLabel)
            make.height.equalTo(18)
        }
        
        applicantNamesLabel.snp.makeConstraints { make in
            make.left.equalTo(18)
            make.right.equalTo(-18)
            make.top.equalTo(titleLabel.snp.bottom).offset(16)
        }
        
        submitTimeLabel.snp.makeConstraints { make in
            make.left.equalTo(18)
            make.right.equalTo(-18)
            make.top.equalTo(applicantNamesLabel.snp.bottom).offset(8)

        }
        
        confirmHKTimeLabel.snp.makeConstraints { make in
            make.left.equalTo(18)
            make.right.equalTo(-18)
            make.top.equalTo(submitTimeLabel.snp.bottom).offset(8)
        }
        
        scheduleView.snp.makeConstraints { make in
            make.left.equalTo(18)
            make.right.equalTo(-18)
            make.top.equalTo(confirmHKTimeLabel.snp.bottom).offset(18)
            make.height.equalTo(84.0)
            make.bottom.equalTo(-18)
        }
    }

    func showBottomCorner(_ isShow: Bool) {
        whiteCornerView.layer.cornerRadius = (isShow ? kCornerRadius6 : 0)
        whiteCornerView.clipsToBounds = true
    }
}
