//
//  YHIncomeInfoEditCell.swift
//  galaxy
//
//  Created by alexzzw on 2024/12/23.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

class YHIncomeInfoEditCell: YHResignDocumentCell {

    static let cellReuseIdentifier = "YHIncomeInfoEditCell"
    private let maxCount = 100
    var didBeginEditing: ((UITextField) -> Void)?
    var didEndEditing: ((UITextField) -> Void)?
    
    private lazy var dotIcon: UIImageView = {
        let imageView = UIImageView(image: UIImage(named: "red_dot_6"))
        return imageView
    }()
    
    private lazy var infoTitleLabel: UILabel = {
        let label = UILabel()
        label.font = .PFSC_R(ofSize: 14)
        label.textColor = UIColor.mainTextColor
        return label
    }()
    
    private lazy var inputTextField: UITextField = {
        let textField = UITextField(frame: CGRect(x: 0, y: 0, width: KScreenWidth, height: 40))
        textField.textColor = .mainTextColor
        textField.font = .PFSC_M(ofSize: 14)
        textField.borderStyle = .none
        textField.delegate = self
        textField.keyboardType = .default
        return textField
    }()
    
    private lazy var bottomLineView: UIView = {
        let view = UIView()
        view.backgroundColor = UIColor(hexString: "#E9ECF0")
        return view
    }()
    
    private lazy var infoFailLabel: UILabel = {
        let label = UILabel()
        label.font = .PFSC_R(ofSize: 12)
        label.textColor = .failColor
        label.text = "请选择"
        return label
    }()
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        setupUI()
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    func setupCellInfo(title: String, detail: String?, placeholder: String = "请输入", showBottomLine: Bool = true, needShowFailButton: Bool, cellType: YHResignRoundCellType = .mid) {
        infoTitleLabel.text = title
        inputTextField.text = detail
        inputTextField.attributedPlaceholder = NSAttributedString(
            string: placeholder,
            attributes: [
                .foregroundColor: UIColor(hex: 0xB9C1CC),
                .font: UIFont.PFSC_R(ofSize: 14)
            ]
        )
        bottomLineView.isHidden = !showBottomLine
        updateFailLabel(needShowFailButton)
        updateCellCorner(cellType)
    }
}

extension YHIncomeInfoEditCell {
    private func updateFailLabel(_ needShowFailButton: Bool) {
        if needShowFailButton {
            infoFailLabel.isHidden = false
            
            infoFailLabel.snp.remakeConstraints { make in
                make.top.equalTo(infoTitleLabel.snp.bottom).offset(6)
                make.left.equalToSuperview().offset(18)
                make.right.lessThanOrEqualToSuperview().offset(-18)
                make.bottom.equalToSuperview().offset(-16)
            }
        } else {
            infoFailLabel.isHidden = true
            
            infoFailLabel.snp.remakeConstraints { make in
                make.left.equalToSuperview().offset(18)
                make.right.lessThanOrEqualToSuperview().offset(-18)
                make.bottom.equalToSuperview().offset(-16)
            }
        }
    }
    
    private func setupUI() {
        updateCellCorner(.mid)
        subContainerView.addSubview(dotIcon)
        subContainerView.addSubview(infoTitleLabel)
        subContainerView.addSubview(inputTextField)
        subContainerView.addSubview(bottomLineView)
        subContainerView.addSubview(infoFailLabel)
        
        dotIcon.setContentCompressionResistancePriority(.required, for: .horizontal)
        infoTitleLabel.setContentCompressionResistancePriority(.defaultHigh, for: .horizontal)
        inputTextField.setContentCompressionResistancePriority(.defaultLow, for: .horizontal)
        
        dotIcon.snp.makeConstraints { make in
            make.centerY.equalToSuperview()
            make.left.equalToSuperview().offset(18)
            make.width.height.equalTo(6)
        }
        
        infoTitleLabel.snp.makeConstraints { make in
            make.top.greaterThanOrEqualToSuperview().offset(16)
            make.bottom.lessThanOrEqualToSuperview().offset(-16)
            make.centerY.equalToSuperview()
            make.left.equalTo(dotIcon.snp.right).offset(2)
        }
        
        inputTextField.snp.makeConstraints { make in
            make.top.greaterThanOrEqualToSuperview()
            make.bottom.lessThanOrEqualToSuperview()
            make.left.equalToSuperview().offset(147)
            make.centerY.equalToSuperview()
            make.right.lessThanOrEqualToSuperview().offset(-18)
        }
        
        bottomLineView.snp.makeConstraints { make in
            make.bottom.equalToSuperview()
            make.left.equalToSuperview().offset(18)
            make.right.equalToSuperview().offset(-18)
            make.height.equalTo(0.5)
        }
        infoFailLabel.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(18)
            make.right.lessThanOrEqualToSuperview().offset(-18)
            make.bottom.equalToSuperview().offset(-16)
        }
        infoFailLabel.isHidden = true
    }
}

extension YHIncomeInfoEditCell: UITextFieldDelegate {
    // MARK: - UITextFieldDelegate
    
    func textFieldDidBeginEditing(_ textField: UITextField) {
        if textField == inputTextField {
            didBeginEditing?(textField)
        }
    }
    
    func textFieldShouldReturn(_ textField: UITextField) -> Bool {
        textField.endEditing(false)
        return true
    }
    
    func textFieldDidEndEditing(_ textField: UITextField) {
        if textField == inputTextField {
            didEndEditing?(textField)
        }
    }
    
    func textField(_ textField: UITextField, shouldChangeCharactersIn range: NSRange, replacementString string: String) -> Bool {
        if textField == inputTextField {
            /// 输入删除
            if string == "" {
                return true
            }
            let count = (textField.text?.utf16.count ?? 0) - range.length + string.utf16.count
            if count > maxCount {
                return false
            }
            return true
        }
        return true
    }
}
