//
//  YHPlayerTransitionAnimator.swift
//  galaxy
//
//  Created by alexzzw on 2024/11/29.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

// class YHPlayerTransitionAnimator: NSObject, UIViewControllerAnimatedTransitioning {
//    enum TransitionType {
//        case zoomIn     // 放大进入
//        case zoomOut    // 缩小退出
//    }
//    
//    private let type: TransitionType
//    private let sourceView: UIView?
//    private let sourceFrame: CGRect?
//    
//    init(type: TransitionType, sourceView: UIView? = nil) {
//        self.type = type
//        self.sourceView = sourceView
//        
//        // 获取源视图在屏幕上的frame
//        if let sourceView = sourceView {
//            self.sourceFrame = sourceView.convert(sourceView.bounds, to: nil)
//        } else {
//            // 默认的浮窗位置和大小
//            self.sourceFrame = CGRect(
//                x: UIScreen.main.bounds.width - 150,
//                y: UIScreen.main.bounds.height - 200,
//                width: 140,
//                height: 80
//            )
//        }
//        
//        super.init()
//    }
//    
//    func transitionDuration(using transitionContext: UIViewControllerContextTransitioning?) -> TimeInterval {
//        return 0.3
//    }
//    
//    func animateTransition(using transitionContext: UIViewControllerContextTransitioning) {
//        guard let toVC = transitionContext.viewController(forKey: .to),
//              let fromVC = transitionContext.viewController(forKey: .from) else {
//            transitionContext.completeTransition(false)
//            return
//        }
//        
//        let containerView = transitionContext.containerView
//        let duration = transitionDuration(using: transitionContext)
//        
//        switch type {
//        case .zoomIn:
//            // 放大进入动画
//            guard let toView = toVC.view else {
//                transitionContext.completeTransition(false)
//                return
//            }
//            
//            containerView.addSubview(toView)
//            
//            // 设置初始状态
//            if let sourceFrame = sourceFrame {
//                toView.frame = sourceFrame
//            } else {
//                toView.transform = CGAffineTransform(scaleX: 0.3, y: 0.3)
//                toView.alpha = 0
//            }
//            
//            // 执行动画
//            UIView.animate(withDuration: duration,
//                         delay: 0,
//                         options: .curveEaseInOut,
//                         animations: {
//                if self.sourceFrame != nil {
//                    toView.frame = UIScreen.main.bounds
//                } else {
//                    toView.transform = .identity
//                    toView.alpha = 1
//                }
//            }, completion: { finished in
//                transitionContext.completeTransition(!transitionContext.transitionWasCancelled)
//            })
//            
//        case .zoomOut:
//            // 缩小退出动画
//            guard let fromView = fromVC.view else {
//                transitionContext.completeTransition(false)
//                return
//            }
//            
//            // 执行动画
//            UIView.animate(withDuration: duration,
//                         delay: 0,
//                         options: .curveEaseInOut,
//                         animations: {
//                if let sourceFrame = self.sourceFrame {
//                    fromView.frame = sourceFrame
//                } else {
//                    fromView.transform = CGAffineTransform(scaleX: 0.3, y: 0.3)
//                    fromView.alpha = 0
//                }
//            }, completion: { finished in
//                fromView.removeFromSuperview()
//                transitionContext.completeTransition(!transitionContext.transitionWasCancelled)
//            })
//        }
//    }
// }

class YHPlayerTransitionAnimator: NSObject, UIViewControllerAnimatedTransitioning {
    enum TransitionType {
        case zoomIn     // 放大进入
        case zoomOut    // 缩小退出
        case push       // 推入
        case pop        // 推出
    }
    
    private let type: TransitionType
    private let sourceView: UIView?
    private let sourceFrame: CGRect?
    
    init(type: TransitionType, sourceView: UIView? = nil) {
        self.type = type
        self.sourceView = sourceView
        
        // 获取源视图在屏幕上的frame
        if let sourceView = sourceView {
            self.sourceFrame = sourceView.convert(sourceView.bounds, to: nil)
        } else {
            // 默认的浮窗位置和大小
            self.sourceFrame = CGRect(
                x: UIScreen.main.bounds.width - 150,
                y: UIScreen.main.bounds.height - 200,
                width: 140,
                height: 80
            )
        }
        
        super.init()
    }
    
    func transitionDuration(using transitionContext: UIViewControllerContextTransitioning?) -> TimeInterval {
        return 0.3
    }
    
    func animateTransition(using transitionContext: UIViewControllerContextTransitioning) {
        guard let toVC = transitionContext.viewController(forKey: .to),
              let fromVC = transitionContext.viewController(forKey: .from) else {
            transitionContext.completeTransition(false)
            return
        }
        
        let containerView = transitionContext.containerView
        let duration = transitionDuration(using: transitionContext)
        let screenBounds = UIScreen.main.bounds
        
        switch type {
        case .zoomIn:
            animateZoomIn(containerView: containerView,
                         fromVC: fromVC,
                         toVC: toVC,
                         duration: duration,
                         transitionContext: transitionContext)
            
        case .zoomOut:
            animateZoomOut(containerView: containerView,
                          fromVC: fromVC,
                          toVC: toVC,
                          duration: duration,
                          transitionContext: transitionContext)
            
        case .push:
            guard let toView = toVC.view else { return }
            containerView.addSubview(toView)
            
            // 设置初始位置（从右边推入）
            toView.frame = screenBounds.offsetBy(dx: screenBounds.width, dy: 0)
            
            UIView.animate(withDuration: duration,
                         delay: 0,
                         options: .curveEaseInOut,
                         animations: {
                // 移动到目标位置
                toView.frame = screenBounds
                fromVC.view.frame = screenBounds.offsetBy(dx: -screenBounds.width * 0.3, dy: 0)
            }, completion: { _ in
                transitionContext.completeTransition(!transitionContext.transitionWasCancelled)
            })
            
        case .pop:
            guard let fromView = fromVC.view,
                  let toView = toVC.view else { return }
            
            // 确保目标视图在底层
            containerView.insertSubview(toView, belowSubview: fromView)
            
            // 设置初始位置
            toView.frame = screenBounds.offsetBy(dx: -screenBounds.width * 0.3, dy: 0)
            
            UIView.animate(withDuration: duration,
                         delay: 0,
                         options: .curveEaseInOut,
                         animations: {
                // 移动到目标位置
                fromView.frame = screenBounds.offsetBy(dx: screenBounds.width, dy: 0)
                toView.frame = screenBounds
            }, completion: { _ in
                fromView.removeFromSuperview()
                transitionContext.completeTransition(!transitionContext.transitionWasCancelled)
            })
        }
    }
    
    private func animateZoomIn(containerView: UIView,
                              fromVC: UIViewController,
                              toVC: UIViewController,
                              duration: TimeInterval,
                              transitionContext: UIViewControllerContextTransitioning) {
        guard let toView = toVC.view else { return }
        containerView.addSubview(toView)
        
        if let sourceFrame = sourceFrame {
            // 有源视图时的放大动画
            toView.frame = sourceFrame
            UIView.animate(withDuration: duration,
                         animations: {
                toView.frame = UIScreen.main.bounds
            }, completion: { _ in
                transitionContext.completeTransition(!transitionContext.transitionWasCancelled)
            })
        } else {
            // 无源视图时执行push动画
            toView.frame = UIScreen.main.bounds.offsetBy(dx: UIScreen.main.bounds.width, dy: 0)
            UIView.animate(withDuration: duration,
                         animations: {
                toView.frame = UIScreen.main.bounds
                fromVC.view.frame = UIScreen.main.bounds.offsetBy(dx: -UIScreen.main.bounds.width * 0.3, dy: 0)
            }, completion: { _ in
                transitionContext.completeTransition(!transitionContext.transitionWasCancelled)
            })
        }
    }
    
    private func animateZoomOut(containerView: UIView,
                               fromVC: UIViewController,
                               toVC: UIViewController,
                               duration: TimeInterval,
                               transitionContext: UIViewControllerContextTransitioning) {
        guard let fromView = fromVC.view else { return }
        
        if let sourceFrame = sourceFrame {
            // 有源视图时的缩小动画
            UIView.animate(withDuration: duration,
                         animations: {
                fromView.frame = sourceFrame
            }, completion: { _ in
                fromView.removeFromSuperview()
                transitionContext.completeTransition(!transitionContext.transitionWasCancelled)
            })
        } else {
            // 无源视图时执行pop动画
            containerView.insertSubview(toVC.view, belowSubview: fromView)
            toVC.view.frame = UIScreen.main.bounds.offsetBy(dx: -UIScreen.main.bounds.width * 0.3, dy: 0)
            
            UIView.animate(withDuration: duration,
                         animations: {
                fromView.frame = UIScreen.main.bounds.offsetBy(dx: UIScreen.main.bounds.width, dy: 0)
                toVC.view.frame = UIScreen.main.bounds
            }, completion: { _ in
                fromView.removeFromSuperview()
                transitionContext.completeTransition(!transitionContext.transitionWasCancelled)
            })
        }
    }
}
