//
//  YHPlayerTopBarView.swift
//  galaxy
//
//  Created by alexzzw on 2024/11/25.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

class YHPlayerTopBarView: UIView {
    private let headerWidth: CGFloat = 32
    private let buttonWidth: CGFloat = 32

    var shareButtonClickEvent: (() -> Void)?
    var zoomButtonClickEvent: (() -> Void)?
    var closeButtonClickEvent: (() -> Void)?
    
    private lazy var bottomVisualEffectView: UIVisualEffectView = {
        let effect = UIBlurEffect(style: .dark)
        let view = UIVisualEffectView(effect: effect)
        view.alpha = 0.9
        return view
    }()

    lazy var headPortrait: UIImageView = {
        let view = UIImageView()
        view.layer.cornerRadius = headerWidth / 2.0
        view.clipsToBounds = true
        view.backgroundColor = .gray
        return view
    }()

    private lazy var infoNameLabel: UILabel = {
        let label = UILabel()
        label.text = "-"
        label.textColor = .white
        label.font = .PFSC_M(ofSize: 14)
        return label
    }()

    private lazy var infoCountLabel: UILabel = {
        let label = UILabel()
        label.text = "-"
        label.textColor = .white.withAlphaComponent(0.7)
        label.font = .PFSC_R(ofSize: 12)
        return label
    }()

    private lazy var shareButton: UIButton = {
        let button = UIButton(type: .custom)
        button.setImage(UIImage(named: "live_room_share"), for: .normal)
        button.addTarget(self, action: #selector(shareButtonClicked), for: .touchUpInside)
        button.contentEdgeInsets = UIEdgeInsets(top: 4, left: 4, bottom: 4, right: 4)
        return button
    }()

    private lazy var zoomButton: UIButton = {
        let button = UIButton(type: .custom)
        button.setImage(UIImage(named: "live_room_zoom"), for: .normal)
        button.addTarget(self, action: #selector(zoomButtonClicked), for: .touchUpInside)
        button.contentEdgeInsets = UIEdgeInsets(top: 4, left: 4, bottom: 4, right: 4)
        return button
    }()

    private lazy var closeButton: UIButton = {
        let button = UIButton(type: .custom)
        button.setImage(UIImage(named: "live_room_close"), for: .normal)
        button.addTarget(self, action: #selector(closeButtonClicked), for: .touchUpInside)
        button.contentEdgeInsets = UIEdgeInsets(top: 4, left: 4, bottom: 4, right: 4)
        return button
    }()

    override init(frame: CGRect) {
        super.init(frame: frame)
        setupUI()
    }

    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    func setupTopBarView(headUrl: String, nickname: String, count: String) {
        if let url = URL(string: headUrl) {
            headPortrait.kf.setImage(with: url, placeholder: UIImage(named: "global_default_image"))
        } else {
            headPortrait.image = UIImage(named: "global_default_image")
        }
        infoNameLabel.text = nickname
        updateCountLabel(count)
    }
    
    func updateCountLabel(_ count: String) {
        infoCountLabel.text = "\(count)人来过"
    }
    
    func hideZoomButton(_ hide: Bool) {
        zoomButton.isHidden = hide
        if hide {
            shareButton.snp.remakeConstraints { make in
                make.centerY.equalTo(headPortrait)
                make.right.equalTo(closeButton.snp.left).offset(-8)
                make.width.height.equalTo(buttonWidth)
                make.left.greaterThanOrEqualTo(infoNameLabel.snp.right).offset(36)
                make.left.greaterThanOrEqualTo(infoCountLabel.snp.right).offset(36)
            }
        } else {
            shareButton.snp.remakeConstraints { make in
                make.centerY.equalTo(headPortrait)
                make.right.equalTo(zoomButton.snp.left).offset(-8)
                make.width.height.equalTo(buttonWidth)
                make.left.greaterThanOrEqualTo(infoNameLabel.snp.right).offset(36)
                make.left.greaterThanOrEqualTo(infoCountLabel.snp.right).offset(36)
            }
        }
    }
}

extension YHPlayerTopBarView {
    @objc private func shareButtonClicked() {
        shareButtonClickEvent?()
    }

    @objc private func zoomButtonClicked() {
        zoomButtonClickEvent?()
    }

    @objc private func closeButtonClicked() {
        closeButtonClickEvent?()
    }

    private func setupUI() {
        addSubview(bottomVisualEffectView)
        addSubview(headPortrait)
        addSubview(infoNameLabel)
        addSubview(infoCountLabel)
        addSubview(shareButton)
        addSubview(zoomButton)
        addSubview(closeButton)
        
        bottomVisualEffectView.frame = CGRect(x: 0, y: 0, width: bounds.size.width, height: bounds.size.height)
        bottomVisualEffectView.snp.makeConstraints { make in
            make.top.left.right.equalToSuperview()
            make.height.equalToSuperview()
        }
        // 渐变效果
        let gradinetLayer = CAGradientLayer()
        gradinetLayer.colors = [UIColor.black.withAlphaComponent(0.16).cgColor, UIColor.black.withAlphaComponent(0.16).cgColor, UIColor.black.withAlphaComponent(0).cgColor, UIColor.black.withAlphaComponent(0).cgColor]
        gradinetLayer.startPoint = CGPoint(x: 0.5, y: 0)
        gradinetLayer.endPoint = CGPoint(x: 0.5, y: 1)
        gradinetLayer.frame = bottomVisualEffectView.bounds
        bottomVisualEffectView.layer.mask = gradinetLayer

        headPortrait.snp.makeConstraints { make in
            make.bottom.equalToSuperview().offset(-6)
            make.left.equalToSuperview().offset(16)
            make.width.height.equalTo(headerWidth)
        }

        infoNameLabel.snp.makeConstraints { make in
            make.top.equalTo(headPortrait.snp.top)
            make.left.equalTo(headPortrait.snp.right).offset(8)
        }

        infoCountLabel.snp.makeConstraints { make in
            make.top.equalTo(infoNameLabel.snp.bottom)
            make.left.equalTo(infoNameLabel.snp.left)
        }

        closeButton.snp.makeConstraints { make in
            make.centerY.equalTo(headPortrait)
            make.right.equalToSuperview().offset(-12)
            make.width.height.equalTo(buttonWidth)
        }

        zoomButton.snp.makeConstraints { make in
            make.centerY.equalTo(headPortrait)
            make.right.equalTo(closeButton.snp.left).offset(-8)
            make.width.height.equalTo(buttonWidth)
        }

        shareButton.snp.makeConstraints { make in
            make.centerY.equalTo(headPortrait)
            make.right.equalTo(zoomButton.snp.left).offset(-8)
            make.width.height.equalTo(buttonWidth)
            make.left.greaterThanOrEqualTo(infoNameLabel.snp.right).offset(36)
            make.left.greaterThanOrEqualTo(infoCountLabel.snp.right).offset(36)
        }
    }
}
