//
//  YHMemberCenterHeaderBannerCell.swift
//  galaxy
//
//  Created by alexzzw on 2025/2/14.
//  Copyright © 2025 https://www.galaxy-immi.com. All rights reserved.
//

import FSPagerView
import UIKit

class YHMemberCenterHeaderBannerCell: FSPagerViewCell {
    var upgradeButtonEvent: ((_ level: YHMembershipLevel) -> Void)?
    var levelButtonEvent: ((_ level: YHMembershipLevel) -> Void)?
    
    private var level: YHMembershipLevel = .normal
    
    private lazy var bannerImagV: UIImageView = {
        let imagV: UIImageView = UIImageView()
        imagV.contentMode = .scaleAspectFit
        imagV.clipsToBounds = true
        imagV.image = UIImage(named: "global_default_image")
        return imagV
    }()

    private lazy var bannerTitleLabel: UILabel = {
        let label = UILabel()
        label.font = UIFont.PFSC_B(ofSize: 24)
        label.textColor = UIColor.white
        return label
    }()

    private lazy var levelButton: UIButton = {
        let button = UIButton(type: .custom)
        button.titleLabel?.font = UIFont.PFSC_R(ofSize: 13)
        button.addTarget(self, action: #selector(levelButtonClicked), for: .touchUpInside)
        return button
    }()

    private lazy var upgradeButton: UIButton = {
        let button = UIButton(type: .custom)
        button.titleLabel?.font = UIFont.PFSC_M(ofSize: 12)
        button.clipsToBounds = true
        button.layer.cornerRadius = 17
        button.addTarget(self, action: #selector(upgradeButtonClicked), for: .touchUpInside)
        return button
    }()

    private lazy var countLabel: UILabel = {
        let label = UILabel()
        label.font = UIFont(name: "DINAlternate-Bold", size: 14)
        label.textColor = UIColor.white
        return label
    }()

    private lazy var progressView: UIProgressView = {
        let view = UIProgressView()
        view.progressTintColor = .white
        view.trackTintColor = UIColor.white.withAlphaComponent(0.16)
        return view
    }()

    private lazy var tipsLabel: UILabel = {
        let label = UILabel()
        label.font = UIFont.PFSC_R(ofSize: 12)
        label.textColor = UIColor.white
        return label
    }()

    override init(frame: CGRect) {
        super.init(frame: frame)
        textLabel?.text = ""
        textLabel?.isHidden = true
        textLabel?.superview?.isHidden = true
        imageView?.isHidden = true

        initView()
    }

    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }

    private func initView() {
        contentView.layer.shadowColor = UIColor.clear.cgColor
        contentView.layer.shadowRadius = 0
        contentView.layer.shadowOpacity = 0
        contentView.layer.shadowOffset = .zero
        contentView.addSubview(bannerImagV)
        contentView.addSubview(bannerTitleLabel)
        contentView.addSubview(levelButton)
        contentView.addSubview(countLabel)
        contentView.addSubview(tipsLabel)
        contentView.addSubview(upgradeButton)
        contentView.addSubview(progressView)
        bannerImagV.setContentCompressionResistancePriority(.required, for: .vertical)
        bannerImagV.setContentCompressionResistancePriority(.required, for: .horizontal)
        bannerImagV.snp.makeConstraints { make in
            make.edges.equalToSuperview()
        }

        bannerTitleLabel.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(24)
            make.top.equalToSuperview().offset(36)
        }

        levelButton.snp.makeConstraints { make in
            make.left.equalTo(bannerTitleLabel)
            make.top.equalTo(bannerTitleLabel.snp.bottom).offset(2)
        }
        countLabel.setContentCompressionResistancePriority(.defaultLow, for: .horizontal)
        progressView.setContentCompressionResistancePriority(.defaultLow, for: .horizontal)
        tipsLabel.setContentCompressionResistancePriority(.defaultLow, for: .horizontal)
        upgradeButton.setContentCompressionResistancePriority(.required, for: .horizontal)
        upgradeButton.snp.makeConstraints { make in
            make.right.equalToSuperview().offset(-26)
            make.bottom.equalToSuperview().offset(-24)
            make.width.equalTo(92)
            make.height.equalTo(34)
        }

        tipsLabel.snp.makeConstraints { make in
            make.bottom.equalTo(upgradeButton)
            make.left.equalTo(bannerTitleLabel)
            make.right.lessThanOrEqualTo(upgradeButton.snp.left).offset(-26)
        }

        let progressViewWidth = KScreenWidth >= 375 ? 158 : 120
        progressView.snp.makeConstraints { make in
            make.left.equalTo(bannerTitleLabel)
            make.bottom.equalTo(tipsLabel.snp.top).offset(-6)
            make.width.equalTo(progressViewWidth)
//            make.right.equalTo(upgradeButton.snp.left).offset(-26)
            make.height.equalTo(2)
        }

        countLabel.snp.makeConstraints { make in
            make.left.equalTo(bannerTitleLabel)
            make.bottom.equalTo(progressView.snp.top).offset(-6)
            make.right.lessThanOrEqualTo(upgradeButton.snp.left).offset(-26)
        }
    }
    
    func setupData(level: YHMembershipLevel, levelString: String, tips: String, upgradeTitle: String, showProgress: Bool, ratioString: String, ratio: CGFloat, showRatio: Bool, showTips: Bool) {
        self.level = level
        bannerImagV.image = UIImage(named: level.cardIconTitle())
        bannerTitleLabel.text = level.description
        bannerTitleLabel.textColor = level.levelTitleColor()
        levelButton.setTitleColor(level.levelButtonTitleColor(), for: .normal)
        levelButton.setTitle(levelString, for: .normal)
        levelButton.setImage(UIImage(named: level.levelButtonImageTitle()), for: .normal)
        levelButton.iconInRight(with: 0)
        countLabel.textColor = level.countTitleColor()
        countLabel.text = ratioString
        countLabel.isHidden = !showRatio
        progressView.progress = Float(ratio)
        tipsLabel.textColor = level.tipsTitleColor()
        tipsLabel.text = tips
        tipsLabel.isHidden = !showTips
        upgradeButton.setTitle(upgradeTitle, for: .normal)
        upgradeButton.setTitleColor(level.upgradeButtonTitleColor(), for: .normal)
        upgradeButton.backgroundColor = level.upgradeButtonBackgroundColor()

        if !showProgress {
            progressView.isHidden = true
            countLabel.snp.remakeConstraints { make in
                make.left.equalTo(bannerTitleLabel)
                make.bottom.equalTo(tipsLabel.snp.top).offset(-6)
                make.right.lessThanOrEqualTo(upgradeButton.snp.left).offset(-26)
            }
        } else {
            progressView.isHidden = false
            countLabel.snp.remakeConstraints { make in
                make.left.equalTo(bannerTitleLabel)
                make.bottom.equalTo(progressView.snp.top).offset(-6)
                make.right.lessThanOrEqualTo(upgradeButton.snp.left).offset(-26)
            }
        }
    }
    
    @objc private func upgradeButtonClicked() {
        upgradeButtonEvent?(level)
    }
    
    @objc private func levelButtonClicked() {
        levelButtonEvent?(level)
    }
}

fileprivate extension YHMembershipLevel {
    func upgradeButtonBackgroundColor() -> UIColor? {
        switch self {
        case .normal:
            return UIColor(hexString: "#323849")
        case .gold:
            return UIColor(hexString: "#5A2C0E")
        case .platinum:
            return UIColor(hexString: "#382F6B")
        case .diamond:
            return UIColor(hexString: "#E0C9AF")
        }
    }

    func upgradeButtonTitleColor() -> UIColor? {
        switch self {
        case .normal:
            return UIColor.white
        case .gold:
            return UIColor.white
        case .platinum:
            return UIColor.white
        case .diamond:
            return UIColor(hexString: "#12296F")
        }
    }

    func levelTitleColor() -> UIColor? {
        switch self {
        case .normal:
            return UIColor.white
        case .gold:
            return UIColor.white
        case .platinum:
            return UIColor.white
        case .diamond:
            return UIColor(hexString: "#EBD7C0")
        }
    }

    func levelButtonTitleColor() -> UIColor? {
        switch self {
        case .normal:
            return UIColor.white.withAlphaComponent(0.5)
        case .gold:
            return UIColor.white.withAlphaComponent(0.5)
        case .platinum:
            return UIColor.white.withAlphaComponent(0.5)
        case .diamond:
            return UIColor(hexString: "#E0C9AF80")?.withAlphaComponent(0.5)
        }
    }
    
    func levelButtonImageTitle() -> String {
        switch self {
        case .normal, .gold, .platinum:
            return "arrow_right_white_16"
        case .diamond:
            return "arrow_right_gold_16"
        }
    }

    func countTitleColor() -> UIColor? {
        switch self {
        case .normal:
            return UIColor.white
        case .gold:
            return UIColor.white
        case .platinum:
            return UIColor.white
        case .diamond:
            return UIColor(hexString: "#E0C9AF")
        }
    }

    func tipsTitleColor() -> UIColor? {
        switch self {
        case .normal:
            return UIColor.white.withAlphaComponent(0.7)
        case .gold:
            return UIColor.white.withAlphaComponent(0.7)
        case .platinum:
            return UIColor.white.withAlphaComponent(0.7)
        case .diamond:
            return UIColor(hexString: "#E0C9AF")
        }
    }
}
