//
//  YHSurveyMatchResultViewController.swift
//  galaxy
//
//  Created by alexzzw on 2025/3/27.
//  Copyright © 2025 https://www.galaxy-immi.com. All rights reserved.
//

import Lottie
import UIKit

class YHSurveyMatchResultViewController: YHBaseViewController {
    enum MatchResult {
        case failure
        case success

        func iconTitle() -> String {
            switch self {
            case .failure:
                return "plan_survey_match_failure"
            case .success:
                return "plan_survey_match_success"
            }
        }

        func title() -> String {
            switch self {
            case .failure:
                return "匹配失败"
            case .success:
                return "匹配完成"
            }
        }

        func content() -> String {
            switch self {
            case .failure:
                return "出了点问题，重试一下吧"
            case .success:
                return "请您查阅【香港身份续签至永居方案】，\n如需进一步咨询，请联系专属顾问和生活管家"
            }
        }

        func buttonTitle() -> String {
            switch self {
            case .failure:
                return "重新匹配"
            case .success:
                return "香港身份续签至永居方案"
            }
        }
    }
    
    private lazy var topImageView: UIImageView = {
        let view = UIImageView()
        return view
    }()

    private lazy var infoTitleLabel: UILabel = {
        let label = UILabel()
        label.textColor = .mainTextColor
        label.font = .PFSC_B(ofSize: 26)
        label.textAlignment = .center
        return label
    }()

    private lazy var infoContentLabel: UILabel = {
        let label = UILabel()
        label.textColor = UIColor(hex: 0x6A7586)
        label.font = .PFSC_R(ofSize: 15)
        label.textAlignment = .center
        label.numberOfLines = 0
        return label
    }()

    private lazy var matchButton: UIButton = {
        let button = UIButton(type: .custom)
        button.addTarget(self, action: #selector(matchButtonClicked), for: .touchUpInside)
        button.backgroundColor = .mainTextColor
        button.setTitleColor(.white, for: .normal)
        button.titleLabel?.font = .PFSC_M(ofSize: 16)
        button.layer.cornerRadius = 3
        button.clipsToBounds = true
        return button
    }()
    
    var matchAgainEvent: (() -> Void)?

    private let pageType: MatchResult
    
    init(_ pageType: MatchResult) {
        self.pageType = pageType
        super.init(nibName: nil, bundle: nil)
    }

    @MainActor required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    override func backItemClick(_ sender: Any) {
        backEvent()
    }

    /// 是否可以返回，包括点击返回和手势返回，默认YES
    override func navigationShouldPop() -> Bool {
        backEvent()
        return false
    }
    
    private func backEvent() {
        if pageType == .success {
            navigationController?.popToRootViewController(animated: true)
        } else {
            matchAgainEvent?()
            navigationController?.popViewController()
        }
    }

    override func viewDidLoad() {
        super.viewDidLoad()
        setupUI()
    }

    private func setupUI() {
        gk_navTitle = "香港身份方案评估"
        gk_navBarAlpha = 1.0
        gk_navBackgroundColor = .white

        view.backgroundColor = UIColor.white
        view.addSubview(topImageView)
        view.addSubview(infoTitleLabel)
        view.addSubview(infoContentLabel)
        view.addSubview(matchButton)
        topImageView.snp.makeConstraints { make in
            make.width.equalTo(200)
            make.height.equalTo(200)
            make.top.equalToSuperview().offset(90 + k_Height_NavigationtBarAndStatuBar)
            make.centerX.equalToSuperview()
        }
        infoTitleLabel.snp.makeConstraints { make in
            make.centerX.equalToSuperview()
            make.top.equalTo(topImageView.snp.bottom).offset(8)
        }
        infoContentLabel.snp.makeConstraints { make in
            make.centerX.equalToSuperview()
            make.top.equalTo(infoTitleLabel.snp.bottom).offset(8)
            make.left.greaterThanOrEqualToSuperview().offset(16)
            make.right.lessThanOrEqualToSuperview().offset(-16)
        }
        matchButton.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(38)
            make.right.equalToSuperview().offset(-38)
            make.bottom.equalTo(view.layoutMarginsGuide.snp.bottom).offset(-68)
            make.height.equalTo(48)
        }
        topImageView.image = UIImage(named: pageType.iconTitle())
        infoTitleLabel.text = pageType.title()
        infoContentLabel.text = pageType.content()
        matchButton.setTitle(pageType.buttonTitle(), for: .normal)
        if pageType == .success {
            playMatchSuccessAnimation()
        }
    }

    @objc private func matchButtonClicked() {
        if pageType == .success {
            guard let navigationController = self.navigationController else {
                return
            }
            let ctl = YHPlanTestViewController()
            ctl.hidesBottomBarWhenPushed = true
            var viewControllers = Array(navigationController.viewControllers.prefix(1))
            viewControllers.append(ctl)
            navigationController.setViewControllers(viewControllers, animated: true)
        } else {
            matchAgainEvent?()
            navigationController?.popViewController()
        }
    }
    
    /// 播放匹配成功
    private func playMatchSuccessAnimation() {
        let aniView = LottieAnimationView(name: "survey_match_success")
        aniView.isUserInteractionEnabled = false
        aniView.frame = CGRect(x: 0, y: 0, width: KScreenWidth, height: KScreenWidth)
        aniView.contentMode = .scaleAspectFit
        aniView.loopMode = .playOnce
        view.addSubview(aniView)
        aniView.play(completion: { _ in
            aniView.removeFromSuperview()
        })
    }
}
