//
//  YHRefreshHeaderAnimator.swift
//  galaxy
//
//  Created by edy on 2024/5/16.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit
import ESPullToRefresh
import YYImage

public extension ES where Base: UIScrollView {
    
    @discardableResult
    func addYHPullToRefresh(handler: @escaping ESRefreshHandler) -> ESRefreshHeaderView {
        return self.addPullToRefresh(animator: YHRefreshHeaderAnimator.init(), handler: handler)
    }
}

class YHRefreshHeaderAnimator: UIView, ESRefreshProtocol, ESRefreshAnimatorProtocol, ESRefreshImpactProtocol {
    
    static let logoWidth = 62.0
    static let logoHeight = 23.0
    static let tipsHeight = 18.0
    static let topMargin = 17.0
    static let bottomMargin = 19.0
    static let midGap = 4.0
    static let headerHeight = YHRefreshHeaderAnimator.topMargin + YHRefreshHeaderAnimator.logoHeight + YHRefreshHeaderAnimator.midGap + YHRefreshHeaderAnimator.tipsHeight + YHRefreshHeaderAnimator.bottomMargin
    
    open var pullToRefreshDescription = "下拉刷新".local {
        didSet {
            if pullToRefreshDescription != oldValue {
                titleLabel.text = pullToRefreshDescription
            }
        }
    }
    open var releaseToRefreshDescription = "松手开始刷新".local
    open var loadingDescription = "加载中...".local

    open var view: UIView { return self }
    open var insets: UIEdgeInsets = UIEdgeInsets.zero
    open var trigger: CGFloat = YHRefreshHeaderAnimator.headerHeight
    open var executeIncremental: CGFloat = YHRefreshHeaderAnimator.headerHeight
    open var state: ESRefreshViewState = .pullToRefresh

    fileprivate let logoImgView: UIImageView = {
        let imageView = UIImageView(image: UIImage(named: "galaxy_icon_refresher"))
        return imageView
    }()
    
    fileprivate let animaitonImgView: YYAnimatedImageView = {
        let img = YYImage(named: "galaxy_refresher.gif")
        let imgV = YYAnimatedImageView(image: img)
        imgV.isHidden = true
        return imgV
    }()
    
    fileprivate let titleLabel: UILabel = {
        let label = UILabel.init(frame: CGRect.zero)
        label.font = UIFont.PFSC_R(ofSize: 12)
        label.textColor = UIColor.init(hex: 0x0046B4, alpha: 0.4)
        label.textAlignment = .left
        return label
    }()
    
    public override init(frame: CGRect) {
        super.init(frame: frame)
        titleLabel.text = pullToRefreshDescription
        self.addSubview(logoImgView)
        self.addSubview(animaitonImgView)
        self.addSubview(titleLabel)
    }
    
    public required init(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    open func refreshAnimationBegin(view: ESRefreshComponent) {
        self.titleLabel.text = loadingDescription
        self.animaitonImgView.isHidden = false
        self.animaitonImgView.currentAnimatedImageIndex = 0
        self.animaitonImgView.startAnimating()
    }
  
    open func refreshAnimationEnd(view: ESRefreshComponent) {
        self.animaitonImgView.isHidden = true
        self.animaitonImgView.stopAnimating()
        self.titleLabel.text = pullToRefreshDescription
    }
    
    open func refresh(view: ESRefreshComponent, progressDidChange progress: CGFloat) {
        // Do nothing
        
    }
    
    open func refresh(view: ESRefreshComponent, stateDidChange state: ESRefreshViewState) {
        guard self.state != state else {
            return
        }
        self.state = state
        
        switch state {
        case .refreshing, .autoRefreshing:
            titleLabel.text = loadingDescription
            self.setNeedsLayout()
            case .releaseToRefresh:
            titleLabel.text = releaseToRefreshDescription
            self.setNeedsLayout()
            self.impact()
            case .pullToRefresh:
            titleLabel.text = pullToRefreshDescription
            self.setNeedsLayout()
        default:
            break
        }
    }

    open override func layoutSubviews() {
        super.layoutSubviews()
        let s = self.bounds.size
        let w = s.width
        
        UIView.performWithoutAnimation {
            titleLabel.sizeToFit()
            logoImgView.frame = CGRect.init(x: 0, y: Self.topMargin, width: Self.logoWidth, height: Self.logoHeight)
            logoImgView.center = CGPoint.init(x: w / 2.0, y: Self.topMargin+Self.logoHeight / 2.0)
            animaitonImgView.frame = logoImgView.frame
            titleLabel.center = CGPoint.init(x: w / 2.0, y: logoImgView.frame.maxY + Self.midGap + Self.tipsHeight/2.0)
        }
    }
    
}
