//
//  YHResourceCategoryView.swift
//  galaxy
//
//  Created by alexzzw on 2025/9/24.
//  Copyright © 2025 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

protocol YHResourceCategoryViewDelegate: AnyObject {
    func categoryView(_ view: YHResourceCategoryView, didSelectCategories categories: [YHResourceCategory], isSet: Bool)
}

class YHResourceCategoryView: UIView {
    
    weak var delegate: YHResourceCategoryViewDelegate?
    
    private var categories: [YHResourceCategory] = []
    private var selectedCategories: [YHResourceCategory] = []
    
    // MARK: - UI Components
    
    private lazy var collectionView: UICollectionView = {
        let layout = UICollectionViewFlowLayout()
        layout.minimumLineSpacing = 8
        layout.minimumInteritemSpacing = 8
        layout.sectionInset = UIEdgeInsets(top: 16, left: 16, bottom: 100, right: 16) // 增加底部间距，为按钮留出空间
        
        let collectionView = UICollectionView(frame: .zero, collectionViewLayout: layout)
        collectionView.backgroundColor = .white
        collectionView.delegate = self
        collectionView.dataSource = self
        collectionView.register(YHResourceCategoryCell.self, forCellWithReuseIdentifier: "YHResourceCategoryCell")
        collectionView.showsVerticalScrollIndicator = false
        return collectionView
    }()
    
    // 底部按钮容器视图
    private lazy var bottomButtonContainer: UIView = {
        let view = UIView()
        view.backgroundColor = .white
        return view
    }()
    
    private lazy var resetButton: UIControl = {
        let btn = UIControl()
        let label = UILabel()
        label.text = "重置"
        label.textColor = .brandGrayColor8
        label.font = UIFont.PFSC_M(ofSize: 12)
        let imageView = UIImageView(image: UIImage(named: "resource_filter_reset"))
        btn.addSubview(label)
        btn.addSubview(imageView)
        label.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(16)
            make.bottom.equalToSuperview()
        }
        imageView.snp.makeConstraints { make in
            make.bottom.equalTo(label.snp.top)
            make.centerX.equalTo(label)
            make.width.height.equalTo(24)
        }
        btn.addTarget(self, action: #selector(resetButtonClicked), for: .touchUpInside)
        return btn
    }()
    
    private lazy var confirmButton: UIButton = {
        let btn = UIButton(type: .custom)
        btn.setTitle("筛选", for: .normal)
        btn.setTitleColor(.white, for: .normal)
        btn.titleLabel?.font = UIFont.PFSC_R(ofSize: 15)
        btn.backgroundColor = UIColor.brandGrayColor8
        btn.layer.cornerRadius = 3
        btn.addTarget(self, action: #selector(confirmButtonClicked), for: .touchUpInside)
        return btn
    }()
    
    // MARK: - Initialization
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        setupUI()
        loadCategories()
    }
    
    required init?(coder: NSCoder) {
        super.init(coder: coder)
        setupUI()
        loadCategories()
    }
    
    // MARK: - Setup
    
    private func setupUI() {
        clipsToBounds = true
        backgroundColor = .white
        
        addSubview(collectionView)
        addSubview(bottomButtonContainer)
        bottomButtonContainer.addSubview(resetButton)
        bottomButtonContainer.addSubview(confirmButton)
        
        setupConstraints()
    }
    
    private func setupConstraints() {
        collectionView.snp.makeConstraints { make in
            make.top.left.right.equalToSuperview()
            make.bottom.equalToSuperview() // collection view占满整个视图
        }
        
        bottomButtonContainer.snp.makeConstraints { make in
            make.left.right.bottom.equalToSuperview()
            make.height.equalTo(74) // 按钮容器高度
        }
        
        resetButton.snp.makeConstraints { make in
            make.left.equalToSuperview()
            make.top.equalToSuperview().offset(12)
            make.width.equalTo(60)
            make.height.equalTo(46)
        }
        
        confirmButton.snp.makeConstraints { make in
            make.left.equalTo(resetButton.snp.right)
            make.right.equalToSuperview().offset(-16)
            make.top.equalToSuperview().offset(12)
            make.height.equalTo(46)
        }
    }
    
    private func loadCategories() {
        // 模拟数据，实际使用时从接口获取
        let allCategory = YHResourceCategory()
        allCategory.id = "0"
        allCategory.name = "全部行业"
        
        let category1 = YHResourceCategory()
        category1.id = "1"
        category1.name = "金融会计"
        
        let category2 = YHResourceCategory()
        category2.id = "2"
        category2.name = "资讯科技"
        
        let category3 = YHResourceCategory()
        category3.id = "3"
        category3.name = "业务支持"
        
        let category4 = YHResourceCategory()
        category4.id = "4"
        category4.name = "工业制造"
        
        let category5 = YHResourceCategory()
        category5.id = "5"
        category5.name = "建筑工程"
        
        let category6 = YHResourceCategory()
        category6.id = "6"
        category6.name = "地产开发"
        
        let category7 = YHResourceCategory()
        category7.id = "7"
        category7.name = "法律服务"
        
        let category8 = YHResourceCategory()
        category8.id = "8"
        category8.name = "商业贸易"
        
        let category9 = YHResourceCategory()
        category9.id = "9"
        category9.name = "物流运输"
        
        let category10 = YHResourceCategory()
        category10.id = "10"
        category10.name = "餐饮旅游"
        
        let category11 = YHResourceCategory()
        category11.id = "11"
        category11.name = "广播娱乐"
        
        let category12 = YHResourceCategory()
        category12.id = "12"
        category12.name = "艺术文化"
        
        let category13 = YHResourceCategory()
        category13.id = "13"
        category13.name = "体育运动"
        
        let category14 = YHResourceCategory()
        category14.id = "14"
        category14.name = "医疗健康"
        
        let category15 = YHResourceCategory()
        category15.id = "15"
        category15.name = "学术教育"
        
        let category16 = YHResourceCategory()
        category16.id = "16"
        category16.name = "其他"
        
        categories = [
            allCategory, category1, category2, category3, category4, category5,
            category6, category7, category8, category9, category10, category11,
            category12, category13, category14, category15, category16
        ]
        
        collectionView.reloadData()
    }
    
    // MARK: - Public Methods
    
    func setSelectedCategories(_ categories: [YHResourceCategory]) {
        selectedCategories = categories
        collectionView.reloadData()
    }
    
    // MARK: - Actions
    
    @objc private func resetButtonClicked() {
        // 重置为"全部行业"
        selectedCategories.removeAll()
        collectionView.reloadData()
        delegate?.categoryView(self, didSelectCategories: selectedCategories, isSet: false)
    }
    
    @objc private func confirmButtonClicked() {
        delegate?.categoryView(self, didSelectCategories: selectedCategories, isSet: true)
    }
}

// MARK: - UICollectionViewDataSource, UICollectionViewDelegate
extension YHResourceCategoryView: UICollectionViewDataSource, UICollectionViewDelegate {
    
    func collectionView(_ collectionView: UICollectionView, numberOfItemsInSection section: Int) -> Int {
        return categories.count
    }
    
    func collectionView(_ collectionView: UICollectionView, cellForItemAt indexPath: IndexPath) -> UICollectionViewCell {
        guard categories.count > indexPath.item, let cell = collectionView.dequeueReusableCell(withReuseIdentifier: "YHResourceCategoryCell", for: indexPath) as? YHResourceCategoryCell else {
            return UICollectionViewCell()
        }
        
        let category = categories[indexPath.item]
        let isSelected = selectedCategories.contains { $0.id == category.id }
        cell.configure(with: category, isSelected: isSelected)
        
        return cell
    }
    
    func collectionView(_ collectionView: UICollectionView, didSelectItemAt indexPath: IndexPath) {
        let category = categories[indexPath.item]
        
        // 如果是"全部行业"，清空其他选择
        if category.id == "0" {
            selectedCategories.removeAll()
            selectedCategories.append(category)
        } else {
            // 移除"全部行业"选项
            selectedCategories.removeAll { $0.id == "0" }
            
            // 切换选中状态
            if let index = selectedCategories.firstIndex(where: { $0.id == category.id }) {
                selectedCategories.remove(at: index)
            } else {
                selectedCategories.append(category)
            }
            
            // 如果没有选中任何分类，自动选中"全部行业"
            if selectedCategories.isEmpty {
                selectedCategories.append(categories.first!)
            }
        }
        
        collectionView.reloadData()
    }
}

// MARK: - UICollectionViewDelegateFlowLayout
extension YHResourceCategoryView: UICollectionViewDelegateFlowLayout {
    
    func collectionView(_ collectionView: UICollectionView, layout collectionViewLayout: UICollectionViewLayout, sizeForItemAt indexPath: IndexPath) -> CGSize {
        // 计算每行三个cell的宽度
        let sectionInset = 16.0 // 左右边距
        let minimumInteritemSpacing = 8.0 // cell间距
        let totalInset = sectionInset * 2 + minimumInteritemSpacing * 2 // 总的边距和间距
        let availableWidth = collectionView.frame.width - totalInset
        let cellWidth = availableWidth / 3.0 // 每行三个
        
        return CGSize(width: cellWidth, height: 44) // 固定高度
    }
}

// MARK: - YHResourceCategoryCell
class YHResourceCategoryCell: UICollectionViewCell {
    
    private lazy var titleLabel: UILabel = {
        let label = UILabel()
        label.font = UIFont.PFSC_R(ofSize: 14)
        label.textAlignment = .center
        label.textColor = .brandGrayColor8
        label.numberOfLines = 1
        label.adjustsFontSizeToFitWidth = true // 自动调整字体大小以适应宽度
        label.minimumScaleFactor = 0.8
        return label
    }()
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        setupUI()
    }
    
    required init?(coder: NSCoder) {
        super.init(coder: coder)
        setupUI()
    }
    
    private func setupUI() {
        backgroundColor = UIColor.brandGrayColor2
        layer.cornerRadius = 2// 调整圆角以适应新高度
        layer.borderWidth = 1
        layer.borderColor = UIColor.clear.cgColor
        
        contentView.addSubview(titleLabel)
        
        titleLabel.snp.makeConstraints { make in
            make.edges.equalToSuperview().inset(UIEdgeInsets(top: 12, left: 4, bottom: 12, right: 4))
        }
    }
    
    func configure(with category: YHResourceCategory, isSelected: Bool) {
        titleLabel.text = category.name
        
        if isSelected {
            layer.borderColor = UIColor.brandGrayColor8.cgColor
            titleLabel.font = UIFont.PFSC_B(ofSize: 14)
        } else {
            layer.borderColor = UIColor.clear.cgColor
            titleLabel.font = UIFont.PFSC_R(ofSize: 14)
        }
    }
}
