//
//  YHResourceViewModel.swift
//  galaxy
//
//  Created by alexzzw on 2025/9/24.
//  Copyright © 2025 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

class YHResourceViewModel: NSObject {
    
    // MARK: - 属性
    var arrResourceData: [YHResourceListModel]?
    var hasMoreForResource: Bool = true
    var currentPage: Int = 1
    var pageSize: Int = 20
    var preloadItemIndex: Int = 10
    
    // 筛选条件
    var currentType: YHResourceFilterType = .service
    var selectedCategories: [YHResourceCategory] = []
    var searchKeyword: String?
    
    override init() {
        super.init()
    }
    
    // MARK: - 网络请求
    
    /// 获取资源列表
    /// - Parameters:
    ///   - firstFlag: 是否首次加载
    ///   - completion: 完成回调
    func getResourceList(firstFlag: Bool, completion: @escaping (Bool, String?) -> Void) {
        
        if firstFlag {
            currentPage = 1
            arrResourceData?.removeAll()
            hasMoreForResource = true
        } else {
            if !hasMoreForResource {
                completion(false, "没有更多数据")
                return
            }
        }
        
        var params: [String: Any] = [
            "page": currentPage,
            "page_size": pageSize,
            "type": currentType == .service ? "service" : "demand"
        ]
        
        // 添加分类筛选（多选）
        if !selectedCategories.isEmpty {
            let categoryIds = selectedCategories.map { $0.id }
            params["category_ids"] = categoryIds
        }
        
        // 添加搜索关键词
        if let keyword = searchKeyword, !keyword.isEmpty {
            params["keyword"] = keyword
        }
        
        // 模拟网络延迟
        DispatchQueue.main.asyncAfter(deadline: .now() + 1.0) {
            let mockData = self.generateMockResourceData(page: self.currentPage)
            
            if firstFlag {
                self.arrResourceData = mockData
            } else {
                self.arrResourceData?.append(contentsOf: mockData)
            }
            
            // 计算每个模型的高度参数
            self.arrResourceData?.forEach { $0.calHeightParam() }
            
            self.currentPage += 1
            self.hasMoreForResource = self.currentPage <= 5 // 模拟5页数据
            
            completion(true, nil)
        }
    }
    
    /// 切换收藏状态
    /// - Parameters:
    ///   - resourceId: 资源ID
    ///   - completion: 完成回调
    func toggleFavorite(resourceId: String, completion: @escaping (Bool, String?) -> Void) {
        // 模拟网络延迟
        DispatchQueue.main.asyncAfter(deadline: .now() + 0.5) {
            if let index = self.arrResourceData?.firstIndex(where: { $0.id == resourceId }) {
                let item = self.arrResourceData![index]
                item.updateFavoriteStatus(!item.is_favorite)
                completion(true, nil)
            } else {
                completion(false, "资源不存在")
            }
        }
    }
    
    /// 获取资源分类
    /// - Parameter completion: 完成回调
    func getResourceCategories(completion: @escaping ([YHResourceCategory]?, String?) -> Void) {
        // 模拟网络延迟
        DispatchQueue.main.asyncAfter(deadline: .now() + 0.5) {
            let categories = self.generateMockCategories()
            completion(categories, nil)
        }
    }
    
    // MARK: - Mock数据生成
    
    private func generateMockResourceData(page: Int) -> [YHResourceListModel] {
        let serviceTitles = [
            "专业财务咨询服务",
            "企业IT系统开发",
            "品牌营销策划方案",
            "法律顾问服务",
            "人力资源管理咨询",
            "企业培训服务",
            "网站建设与维护",
            "财务审计服务"
        ]
        
        let demandTitles = [
            "寻找优质供应商",
            "招聘高级技术人员",
            "需要市场推广合作",
            "寻求投资合作伙伴",
            "需要办公场地租赁",
            "寻找物流配送服务",
            "需要设备采购咨询",
            "寻求技术合作"
        ]
        
        let companies = [
            ("北京科技有限公司哈哈哈哈哈哈哈哈哈哈哈哈哈哈哈哈哈哈哈哈哈", "https://picsum.photos/60/60?random=1"),
            ("上海贸易集团", "https://picsum.photos/60/60?random=2"),
            ("深圳创新科技", "https://picsum.photos/60/60?random=3"),
            ("杭州电商公司", "https://picsum.photos/60/60?random=4"),
            ("成都软件开发", "https://picsum.photos/60/60?random=5"),
            ("广州制造企业", "https://picsum.photos/60/60?random=6"),
            ("武汉咨询公司", "https://picsum.photos/60/60?random=7"),
            ("西安技术公司", "https://picsum.photos/60/60?random=8")
        ]
        
        let categories = [
            ("finance", "金融会计哈哈哈哈哈哈"),
            ("tech", "资讯科技"),
            ("business", "业务支援"),
            ("manufacturing", "工业制造"),
            ("legal", "法律服务"),
            ("trade", "商业贸易"),
            ("logistics", "物流运输"),
            ("other", "其他")
        ]
        
        let contacts = [
            ("张经理", "138****1234", "zhang123"),
            ("李总监", "139****5678", "li_manager"),
            ("王助理", "136****9876", "wang_assistant"),
            ("赵主管", "137****4321", "zhao_supervisor"),
            ("孙经理", "135****8765", "sun_manager"),
            ("周总", "134****2468", "zhou_boss"),
            ("吴主任", "133****1357", "wu_director"),
            ("郑经理", "132****9753", "zheng_manager")
        ]
        
        let locations = ["北京·朝阳区", "上海·浦东新区", "深圳·南山区", "杭州·西湖区", "成都·锦江区", "广州·天河区", "武汉·江汉区", "西安·雁塔区"]
        
        let contents = [
            "我们提供专业的财务咨询服务，帮助企业优化财务结构，降低运营成本，提高资金使用效率。团队拥有10年以上行业经验。",
            "专业的IT系统开发团队，提供定制化解决方案，包括Web应用、移动应用、企业管理系统等。技术栈涵盖主流开发框架。",
            "专注品牌营销策划，提供全方位的品牌推广方案。从品牌定位到执行落地，一站式服务，已服务500+企业客户。",
            "资深律师团队提供专业法律顾问服务，涵盖合同审查、法律风险评估、争议解决等。保障企业合规经营。",
            "人力资源管理专家，提供招聘、培训、绩效管理等全方位HR服务。帮助企业建立完善的人才管理体系。",
            "企业培训专业机构，提供管理培训、技能培训、团队建设等服务。定制化课程设计，提升员工综合素质。",
            "专业网站建设团队，提供网站设计、开发、运维一体化服务。响应式设计，SEO优化，提升企业网络形象。",
            "注册会计师团队提供财务审计服务，确保财务报表真实可靠。严格按照审计准则执行，保障审计质量。"
        ]
        
        // 根据筛选条件过滤数据
        var filteredData: [YHResourceListModel] = []
        
        for i in 0..<8 {
            let model = YHResourceListModel()
            let baseId = (page - 1) * 8 + i
            
            model.id = "resource_\(baseId)"
            
            // 根据当前类型选择标题
            let titles = currentType == .service ? serviceTitles : demandTitles
            model.title = titles[i % titles.count]
            model.content = contents[i % contents.count]
            model.type = currentType == .service ? "service" : "demand"
            
            // 公司信息
            let company = companies[i % companies.count]
            model.company_name = company.0
            model.company_logo = company.1
            
            // 分类信息
            let category = categories[i % categories.count]
            model.category_id = category.0
            model.category_name = category.1
            
            // 联系信息
            let contact = contacts[i % contacts.count]
            model.contact_name = contact.0
            model.contact_phone = contact.1
            model.contact_wechat = contact.2
            
            // 其他信息
            model.location = locations[i % locations.count]
            model.price = i % 3 == 0 ? "" : "\(Int.random(in: 1000...9999))"
            model.price_unit = i % 3 == 0 ? "" : ["小时", "天", "月", "项目"][i % 4]
            model.view_count = Int.random(in: 10...999)
            model.favorite_count = Int.random(in: 0...50)
            model.is_favorite = Bool.random()
            model.status = 1 // 已发布
            model.certification_status = i % 2 // 随机认证状态
            
            // 时间
            let daysAgo = Int.random(in: 0...30)
            let date = Calendar.current.date(byAdding: .day, value: -daysAgo, to: Date()) ?? Date()
            let formatter = DateFormatter()
            formatter.dateFormat = "yyyy-MM-dd HH:mm:ss"
            model.created_time = formatter.string(from: date)
            
            // 图片（随机添加1-3张图片）
            let imageCount = Int.random(in: 0...3)
            model.images = (0..<imageCount).map { "https://picsum.photos/300/200?random=\(baseId)_\($0)" }
            
            // 标签
            let allTags = ["优质服务", "专业团队", "价格优惠", "经验丰富", "快速响应", "全程跟踪", "售后保障", "行业领先"]
            let tagCount = Int.random(in: 0...3)
            model.tags = Array(allTags.shuffled().prefix(tagCount))
            
            // 应用筛选条件
            var shouldInclude = true
            
            // 分类筛选
            if !selectedCategories.isEmpty {
                let categoryIds = selectedCategories.map { $0.id }
                shouldInclude = shouldInclude && categoryIds.contains(model.category_id)
            }
            
            // 搜索关键词筛选
            if let keyword = searchKeyword, !keyword.isEmpty {
                let searchText = "\(model.title) \(model.content) \(model.company_name)".lowercased()
                shouldInclude = shouldInclude && searchText.contains(keyword.lowercased())
            }
            
            if shouldInclude {
                filteredData.append(model)
            }
        }
        
        return filteredData
    }
    
    private func generateMockCategories() -> [YHResourceCategory] {
        let categoryData = [
            ("finance", "金融会计", "💰"),
            ("tech", "资讯科技", "💻"),
            ("business", "业务支援", "📊"),
            ("manufacturing", "工业制造", "🏭"),
            ("construction", "建筑工程", "🏗️"),
            ("real_estate", "地产开发", "🏢"),
            ("legal", "法律服务", "⚖️"),
            ("trade", "商业贸易", "🛒"),
            ("logistics", "物流运输", "🚚"),
            ("food", "餐饮旅游", "🍽️"),
            ("media", "广播娱乐", "📺"),
            ("culture", "艺术文化", "🎨"),
            ("sports", "体育运动", "⚽"),
            ("health", "医疗健康", "🏥"),
            ("education", "学术教育", "📚"),
            ("other", "其他", "📋")
        ]
        
        return categoryData.map { (id, name, icon) in
            let category = YHResourceCategory()
            category.id = id
            category.name = name
            category.icon = icon
            category.resource_count = Int.random(in: 10...999)
            category.is_hot = Int.random(in: 0...1) == 1
            return category
        }
    }
}
