//
//  YHInfoQuestionSelectionCell.swift
//  galaxy
//
//  Created by alexzzw on 2024/12/21.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

class YHInfoQuestionSelectionCell: YHResignDocumentCell {
    
    static let cellReuseIdentifier = "YHInfoQuestionSelectionCell"
    var didSelectSureButton: ((YHCheckboxSelectType) -> Void)?
    
    var selectedSureButton: YHCheckboxSelectType = .unknown {
        didSet {
            updateButtonsState(selectedSureButton)
        }
    }

    private lazy var dotIcon: UIImageView = {
        let imageView = UIImageView(image: UIImage(named: "red_dot_6"))
        return imageView
    }()
    
    private lazy var infoQuestionLabel: UILabel = {
        let label = UILabel()
        label.font = .PFSC_R(ofSize: 14)
        label.textColor = UIColor.mainTextColor
        label.numberOfLines = 0
        label.lineBreakMode = .byWordWrapping
        return label
    }()
    
    private lazy var sureButton: UIButton = {
        let button = UIButton(type: .custom)
        button.titleLabel?.font = .PFSC_M(ofSize: 13)
        button.setTitle("是", for: .normal)
        button.setTitleColor(.mainTextColor, for: .normal)
        button.setTitleColor(.brandMainColor, for: .selected)
        button.addTarget(self, action: #selector(sureButtonDidClick(_:)), for: .touchUpInside)
        button.layer.cornerRadius = 16.0
        button.clipsToBounds = true
        return button
    }()
    
    private lazy var cancelButton: UIButton = {
        let button = UIButton(type: .custom)
        button.titleLabel?.font = .PFSC_M(ofSize: 13)
        button.setTitle("否", for: .normal)
        button.setTitleColor(.mainTextColor, for: .normal)
        button.setTitleColor(.brandMainColor, for: .selected)
        button.addTarget(self, action: #selector(cancelButtonDidClick(_:)), for: .touchUpInside)
        button.layer.cornerRadius = 16
        button.clipsToBounds = true
        return button
    }()
    
    private lazy var infoFailLabel: UILabel = {
        let label = UILabel()
        label.font = .PFSC_R(ofSize: 12)
        label.textColor = .failColor
        label.text = "请选择"
        return label
    }()
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        setupUI()
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    func setupCellInfo(question: String, showBottomLine: Bool = false, needShowFailButton: Bool) {
        infoQuestionLabel.text = question
        updateFailLabel(needShowFailButton)
    }

}

extension YHInfoQuestionSelectionCell {
    private func updateFailLabel(_ needShowFailButton: Bool) {
        if needShowFailButton {
            infoFailLabel.isHidden = false
            infoQuestionLabel.snp.remakeConstraints { make in
                make.left.equalTo(dotIcon.snp.right).offset(2)
                make.top.equalToSuperview().offset(16)
                make.height.greaterThanOrEqualTo(40)
                make.width.lessThanOrEqualTo(113)
            }
            infoFailLabel.snp.remakeConstraints { make in
                make.top.equalTo(infoQuestionLabel.snp.bottom).offset(6)
                make.left.equalToSuperview().offset(18)
                make.right.lessThanOrEqualToSuperview().offset(-18)
                make.bottom.equalToSuperview().offset(-16)
            }
        } else {
            infoFailLabel.isHidden = true
            infoQuestionLabel.snp.remakeConstraints { make in
                make.left.equalTo(dotIcon.snp.right).offset(2)
                make.top.equalToSuperview().offset(16)
                make.height.greaterThanOrEqualTo(40)
                make.width.lessThanOrEqualTo(113)
                make.bottom.equalToSuperview().offset(-16)
            }
            infoFailLabel.snp.remakeConstraints { make in
                make.left.equalToSuperview().offset(18)
                make.right.lessThanOrEqualToSuperview().offset(-18)
                make.bottom.equalToSuperview().offset(-16)
            }
        }
    }
    
    @objc private func cancelButtonDidClick(_ sender: UIButton) {
        guard !sender.isSelected else {
            return
        }
        selectedSureButton = .false
        didSelectSureButton?(.false)
    }

    @objc private func sureButtonDidClick(_ sender: UIButton) {
        guard !sender.isSelected else {
            return
        }
        selectedSureButton = .true
        didSelectSureButton?(.true)
    }

    private func setupUI() {
        updateCellCorner(.mid)
        subContainerView.addSubview(dotIcon)
        subContainerView.addSubview(infoQuestionLabel)
        subContainerView.addSubview(sureButton)
        subContainerView.addSubview(cancelButton)
        subContainerView.addSubview(infoFailLabel)

        sureButton.setContentCompressionResistancePriority(.required, for: .horizontal)
        cancelButton.setContentCompressionResistancePriority(.required, for: .horizontal)
        infoQuestionLabel.setContentCompressionResistancePriority(.defaultLow, for: .horizontal)

        dotIcon.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(18)
            make.height.width.equalTo(6)
            make.top.equalTo(infoQuestionLabel.snp.top).offset(7)
        }

        infoQuestionLabel.snp.makeConstraints { make in
            make.left.equalTo(dotIcon.snp.right).offset(2)
            make.top.equalToSuperview().offset(16)
            make.height.greaterThanOrEqualTo(40)
            make.width.lessThanOrEqualTo(113)
            make.bottom.equalToSuperview().offset(-16)
        }

        cancelButton.snp.makeConstraints { make in
            make.right.equalToSuperview().offset(-18)
            make.height.equalTo(32)
            make.width.equalTo(69)
            make.centerY.equalTo(infoQuestionLabel)
        }

        sureButton.snp.makeConstraints { make in
            make.right.equalTo(cancelButton.snp.left).offset(-8)
            make.height.equalTo(32)
            make.width.equalTo(69)
            make.centerY.equalTo(infoQuestionLabel)
            make.left.greaterThanOrEqualTo(infoQuestionLabel.snp.right).offset(8)
        }

        infoFailLabel.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(18)
            make.right.lessThanOrEqualToSuperview().offset(-18)
            make.bottom.equalToSuperview().offset(-16)
        }
        infoFailLabel.isHidden = true
        selectedSureButton = .unknown
    }

    private func updateButtonsState(_ state: YHCheckboxSelectType) {
        switch state {
        case .unknown:
            sureButton.isSelected = false
            sureButton.backgroundColor = .contentBkgColor
            sureButton.layer.borderColor = nil
            sureButton.layer.borderWidth = 0
            cancelButton.isSelected = false
            cancelButton.backgroundColor = .contentBkgColor
            cancelButton.layer.borderColor = nil
            cancelButton.layer.borderWidth = 0
        case .true:
            sureButton.isSelected = true
            sureButton.backgroundColor = UIColor(hexString: "#EBF0F9")
            sureButton.layer.borderColor = UIColor.brandMainColor.cgColor
            sureButton.layer.borderWidth = 1
            
            cancelButton.isSelected = false
            cancelButton.backgroundColor = .contentBkgColor
            cancelButton.layer.borderColor = nil
            cancelButton.layer.borderWidth = 0
        case .false:
            cancelButton.isSelected = true
            cancelButton.backgroundColor = UIColor(hexString: "#EBF0F9")
            cancelButton.layer.borderColor = UIColor.brandMainColor.cgColor
            cancelButton.layer.borderWidth = 1
            sureButton.isSelected = false
            sureButton.backgroundColor = .contentBkgColor
            sureButton.layer.borderColor = nil
            sureButton.layer.borderWidth = 0
        }
    }

}
