//
//  YHGCApplicationTypeResultController.swift
//  galaxy
//
//  Created by alexzzw on 2024/11/11.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

class YHGCApplicationTypeResultController: YHBaseViewController {
    private lazy var bgIcon: UIImageView = {
        let view = UIImageView()
        return view
    }()
    
    private lazy var backButton: UIButton = {
        let button = UIButton(type: .custom)
        button.titleLabel?.font = UIFont.PFSC_M(ofSize: 16)
        button.setTitle("返回", for: .normal)
        button.setTitle("返回", for: .highlighted)
        button.setTitleColor( UIColor(hex: 0xffffff), for: .normal)
        button.setTitleColor( UIColor(hex: 0xffffff), for: .highlighted)
        button.addTarget(self, action: #selector(clickBackButton), for: .touchUpInside)
        button.layer.cornerRadius = kCornerRadius3
        button.clipsToBounds = true
        button.backgroundColor = .brandMainColor
        return button
    }()
    
    private lazy var headerLabel: UILabel = {
        let label = UILabel()
        label.textColor = .mainTextColor
        label.font = .PFSC_M(ofSize: 21)
        label.text = "您已选定高才申请类型"
        return label
    }()
    
    private lazy var infoTitleLabel: UILabel = {
        let label = UILabel()
        label.font = .PFSC_B(ofSize: 17)
        label.textColor = .mainTextColor
        return label
    }()
    
    private lazy var infoDetailLabel: UILabel = {
        let label = UILabel()
        label.textColor = UIColor(hexString: "#6D788A")
        label.font = .PFSC_R(ofSize: 13)
        label.lineBreakMode = .byCharWrapping
        label.numberOfLines = 3
        return label
    }()
    
    private lazy var barView: UIView = {
        let view = UIView()
        return view
    }()
    
    private lazy var actionBtn: UIButton = {
        let button = UIButton(type: .custom)
        button.setTitle("申请标准", for: .normal)
        button.setTitleColor(.mainTextColor, for: .normal)
        button.titleLabel?.font = .PFSC_R(ofSize: 14)
        button.setImage(UIImage(named: "right_arrow_black_20"), for: .normal)
        button.addTarget(self, action: #selector(actionBtnClicked), for: .touchUpInside)
        return button
    }()
    
    private lazy var subContainerView: YHBaseDynamicCornerRadiusView = {
        let view = YHBaseDynamicCornerRadiusView(cornerRadius: 6, corner: .allCorners)
        view.backgroundColor = .white
        return view
    }()
    
    private lazy var bottomView: UIView = {
        let view = UIView()
        view.backgroundColor = .white
        return view
    }()
        
    var backButtonEvent: (() -> Void)?
    
    private let type: YHGCApplicationType
    init(type: YHGCApplicationType) {
        self.type = type
        super.init(nibName: nil, bundle: nil)
    }
    
    @MainActor required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    override func viewDidLoad() {
        super.viewDidLoad()
        setupUI()
    }
}

extension YHGCApplicationTypeResultController {
    
    @objc private func actionBtnClicked() {
        showPopVC(type: type)
    }
    
    @objc private func clickBackButton() {
        backButtonEvent?()
        navigationController?.popViewController(animated: true)
    }
    
    private func setupUI() {
        gk_navTitle = "申请类别"
        gk_navBarAlpha = 0
        gk_navigationBar.backgroundColor = .clear
        view.backgroundColor = UIColor.contentBkgColor
        view.addSubview(bgIcon)
        view.addSubview(headerLabel)
        view.addSubview(bottomView)
        bottomView.addSubview(backButton)
        view.addSubview(subContainerView)
        subContainerView.addSubview(barView)
        subContainerView.addSubview(infoTitleLabel)
        subContainerView.addSubview(infoDetailLabel)
        subContainerView.addSubview(actionBtn)
        
        let ratio = 318.0 / 375.0
        bgIcon.snp.makeConstraints { make in
            make.top.left.right.equalToSuperview()
            make.height.equalTo(bgIcon.snp.width).multipliedBy(ratio)
        }
        
        headerLabel.snp.makeConstraints { make in
            make.top.equalTo(k_Height_NavigationtBarAndStatuBar + 42)
            make.left.equalToSuperview().offset(20)
            make.right.lessThanOrEqualToSuperview().offset(-20)
        }
        
        subContainerView.snp.makeConstraints { make in
            make.top.equalTo(headerLabel.snp.bottom).offset(50)
            make.left.equalToSuperview().offset(20)
            make.right.equalToSuperview().offset(-20)
            make.height.equalTo(148)
        }
        
        barView.snp.makeConstraints { make in
            make.left.equalToSuperview()
            make.centerY.equalToSuperview()
            make.width.equalTo(3)
            make.height.equalTo(108)
        }
        
        infoTitleLabel.snp.makeConstraints { make in
            make.left.equalTo(barView.snp.right).offset(20)
            make.top.equalTo(barView.snp.top)
            make.right.lessThanOrEqualToSuperview().offset(-20)
        }
        
        infoDetailLabel.snp.makeConstraints { make in
            make.left.equalTo(infoTitleLabel.snp.left)
            make.top.equalTo(infoTitleLabel.snp.bottom).offset(8)
            make.right.lessThanOrEqualToSuperview().offset(-20)
        }
        
        actionBtn.snp.makeConstraints { make in
            make.left.equalTo(infoTitleLabel.snp.left)
            make.top.greaterThanOrEqualTo(infoDetailLabel.snp.bottom).offset(0).priority(.high)
            make.bottom.equalTo(barView.snp.bottom)
        }
        
        actionBtn.iconInRight(with: 0)
        
        bottomView.snp.makeConstraints { make in
            make.left.right.bottom.equalToSuperview()
            make.top.equalTo(view.safeAreaLayoutGuide.snp.bottom).offset(-66)
        }
        
        backButton.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(kMargin)
            make.right.equalToSuperview().offset(-kMargin)
            make.top.equalToSuperview().offset(8)
            make.height.equalTo(48)
        }
        bgIcon.image = UIImage(named: type.bgIconTitle())
        infoTitleLabel.text = type.titleString
        infoDetailLabel.text = type.detailString
        barView.backgroundColor = type.barColor()
    }
    
    private func showPopVC(type: YHGCApplicationType) {
        let vc = YHGCVisaProgramPopVC(type: type)
        let pc = YHBottomPresentationController(presentedViewController: vc, presenting: self)
        pc.customRadius = 8.0
        vc.transitioningDelegate = pc
        vc.sureButtonEvent = { [weak vc] in
            vc?.dismiss(animated: true)
        }
        present(vc, animated: true, completion: nil)
    }
}

private extension YHGCApplicationType {
    func bgIconTitle() -> String {
        switch self {
        case .typeA:
            return "gc_application_type_bg_a"
        case .typeB:
            return "gc_application_type_bg_b"
        case .typeC:
            return "gc_application_type_bg_c"
        }
    }
    
    func barColor() -> UIColor? {
        switch self {
        case .typeA:
            return UIColor(hexString: "#EEDBBD")
        case .typeB:
            return UIColor(hexString: "#BFCDEF")
        case .typeC:
            return UIColor(hexString: "#BBE1F2")
        }
    }
}
