//
//  YHResourceDetailViewController.swift
//  galaxy
//
//  Created by alexzzw on 2025/9/24.
//  Copyright © 2025 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit
import SnapKit

class YHResourceDetailViewController: YHBaseViewController {
    
    // MARK: - Properties
    var resourceModel: YHResourceListModel?
    
    var isMyCompany: Bool = false {
        didSet {
            updateViewIsMine()
        }
    }
    
    private lazy var messageHandler: YHIMMessageHandler = {
        return YHIMMessageHandler(
            viewController: self,
            navigationController: navigationController
        )
    }()
    
    // MARK: - UI Components
    
    // 背景图 - 改为在tableView后面
    private lazy var bgIcon: UIImageView = {
        let view = UIImageView()
        view.image = UIImage(named: "resource_detail_bg")
        view.contentMode = .scaleAspectFill
        view.clipsToBounds = true
        return view
    }()
    
    private lazy var tableView: UITableView = {
        let view = UITableView(frame: CGRect(x: 0, y: 0, width: KScreenWidth, height: KScreenHeight), style: .grouped)
        view.estimatedSectionHeaderHeight = 0.01
        view.estimatedSectionFooterHeight = 0.01
        view.contentInsetAdjustmentBehavior = .never
        view.backgroundColor = .clear
        view.separatorStyle = .none
        view.dataSource = self
        view.delegate = self
        view.showsVerticalScrollIndicator = false
        view.register(YHResourceDetailHeaderCell.self, forCellReuseIdentifier: "HeaderCell")
        view.register(YHResourceDetailBaseInfoCell.self, forCellReuseIdentifier: "BaseInfoCell")
        view.register(YHResourceDetailDemandCell.self, forCellReuseIdentifier: "DemandCell")
        view.register(YHResourceDetailDocumentsCell.self, forCellReuseIdentifier: "DocumentsCell")
        return view
    }()
    
    private lazy var bottomActionView: UIView = {
        let view = UIView()
        view.backgroundColor = .white
        return view
    }()
    
    private lazy var contactButton: UIButton = {
        let button = UIButton(type: .custom)
        button.titleLabel?.font = UIFont.PFSC_M(ofSize: 16)
        button.setTitle("联系合作", for: .normal)
        button.setTitleColor(UIColor.white, for: .normal)
        button.backgroundColor = UIColor.brandGrayColor8
        button.addTarget(self, action: #selector(contactButtonClicked), for: .touchUpInside)
        return button
    }()
    
    // MARK: - Lifecycle
    
    override func viewDidLoad() {
        super.viewDidLoad()
        setupUI()
        loadData()
    }
    
    // MARK: - Setup
    
    private func setupUI() {
        gk_navBarAlpha = 0
        gk_navBackgroundColor = .clear
        gk_backImage = UIImage(named: "nav_black_24")
        view.addSubview(bgIcon)
        view.addSubview(tableView)
        // 添加底部操作栏
        view.addSubview(bottomActionView)
        bottomActionView.addSubview(contactButton)
        setupConstraints()
        updateViewIsMine()
    }
    
    private func updateViewIsMine() {
        gk_navTitle = isMyCompany ? "我的企业" : "企业详情"
        if isMyCompany {
            bottomActionView.isHidden = true
        } else {
            bottomActionView.isHidden = false
        }
        updateRightBarButtonItem()
    }
    
    private func setupConstraints() {
        // 背景图约束 - 参照YHMemberCenterViewController
        bgIcon.snp.makeConstraints { make in
            make.top.left.right.bottom.equalToSuperview()
        }
        
        // tableView约束
        tableView.snp.makeConstraints { make in
            make.left.right.equalToSuperview()
            make.top.equalTo(k_Height_NavigationtBarAndStatuBar)
            make.bottom.equalTo(bottomActionView.snp.top)
        }
        
        bottomActionView.snp.makeConstraints { make in
            make.left.right.equalToSuperview()
            make.bottom.equalToSuperview()
            make.top.equalTo(view.safeAreaLayoutGuide.snp.bottom).offset(-64)
        }
        
        contactButton.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(16)
            make.right.equalToSuperview().offset(-16)
            make.top.equalToSuperview().offset(8)
            make.height.equalTo(46)
        }
        
        // 不需要设置contentInset，让内容从导航栏下方开始
        tableView.tableFooterView = UIView(frame: CGRect(x: 0, y: 0, width: KScreenWidth, height: 0.01))
    }
    
    private func updateRightBarButtonItem() {
        if isMyCompany {
            let rightButtonItem = UIBarButtonItem(
                image: UIImage(named: "service_center_more")?.withRenderingMode(.alwaysOriginal),
                style: .plain,
                target: self,
                action: #selector(moreButtonClicked)
            )
            gk_navRightBarButtonItem = rightButtonItem
            gk_navItemRightSpace = 16
        } else {
            let rightButtonItem = UIBarButtonItem(
                image: UIImage(named: "share_item")?.withRenderingMode(.alwaysOriginal),
                style: .plain,
                target: self,
                action: #selector(shareButtonClicked)
            )
            gk_navRightBarButtonItem = rightButtonItem
            gk_navItemRightSpace = 16
        }
        
    }
    
    // MARK: - Data
    
    private func loadData() {
        guard let _ = resourceModel else { return }
        tableView.reloadData()
        incrementViewCount()
    }
    
    private func incrementViewCount() {
        resourceModel?.incrementViewCount()
    }
    
    // MARK: - Actions
    
    @objc private func shareButtonClicked() {
        //
    }
    
    @objc private func moreButtonClicked() {
        //
    }
    
    @objc private func contactButtonClicked() {
        if !YHLoginManager.shared.isLogin() {
            YHOneKeyLoginManager.shared.oneKeyLogin()
            return
        }
        messageHandler.gotoChatVC(senderID: "")
    }
    
}

// MARK: - UITableViewDataSource & UITableViewDelegate

extension YHResourceDetailViewController: UITableViewDataSource, UITableViewDelegate {
    
    func numberOfSections(in tableView: UITableView) -> Int {
        return 4
    }
    
    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return 1
    }
    
    func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        guard let model = resourceModel else {
            return UITableViewCell()
        }
        
        let hasDocuments = !model.images.isEmpty
        
        switch indexPath.section {
        case 0:
            guard let cell = tableView.dequeueReusableCell(withIdentifier: "HeaderCell", for: indexPath) as? YHResourceDetailHeaderCell else {
                return UITableViewCell()
            }
            cell.configure(with: model)
            return cell
            
        case 1:
            guard let cell = tableView.dequeueReusableCell(withIdentifier: "BaseInfoCell", for: indexPath) as? YHResourceDetailBaseInfoCell else {
                return UITableViewCell()
            }
            cell.configure(with: model)
            return cell
            
        case 2:
            if hasDocuments {
                guard let cell = tableView.dequeueReusableCell(withIdentifier: "DemandCell", for: indexPath) as? YHResourceDetailDemandCell else {
                    return UITableViewCell()
                }
                cell.configure(with: model)
                return cell
            } else {
                guard let cell = tableView.dequeueReusableCell(withIdentifier: "DemandCell", for: indexPath) as? YHResourceDetailDemandCell else {
                    return UITableViewCell()
                }
                cell.configure(with: model)
                return cell
            }
            
        case 3:
            guard let cell = tableView.dequeueReusableCell(withIdentifier: "DocumentsCell", for: indexPath) as? YHResourceDetailDocumentsCell else {
                return UITableViewCell()
            }
            cell.configure(with: model.images)
            cell.onImageTapped = { [weak self] index in
                self?.showImageBrowser(at: index)
            }
            return cell
            
        default:
            return UITableViewCell()
        }
    }
    
    func tableView(_ tableView: UITableView, heightForRowAt indexPath: IndexPath) -> CGFloat {
        return UITableView.automaticDimension
    }
    
    func tableView(_ tableView: UITableView, viewForHeaderInSection section: Int) -> UIView? {
        return UIView()
    }
    
    func tableView(_ tableView: UITableView, viewForFooterInSection section: Int) -> UIView? {
        return UIView()
    }
    
    func tableView(_ tableView: UITableView, heightForHeaderInSection section: Int) -> CGFloat {
        return 0.01
    }
    
    func tableView(_ tableView: UITableView, heightForFooterInSection section: Int) -> CGFloat {
        return 0.01
    }
    
    // 关键改动：参照YHMemberCenterViewController的scrollViewDidScroll实现
    func scrollViewDidScroll(_ scrollView: UIScrollView) {
        if scrollView.contentOffset.y > 0 {
            // 向上滚动
            var alpha = scrollView.contentOffset.y / k_Height_NavigationtBarAndStatuBar
            if alpha > 1.0 {
                alpha = 1.0
            }
            gk_navBarAlpha = alpha
            gk_navBackgroundColor = .white
            gk_navTitle = isMyCompany ? "我的企业" : "企业详情"
        } else {
            gk_navBarAlpha = 0.0
            gk_navBackgroundColor = .clear
            gk_navTitle = nil
        }
    }
    
    private func showImageBrowser(at index: Int) {
        guard let images = resourceModel?.images else { return }
        print("显示图片浏览器，索引：\(index)，共\(images.count)张")
    }
}
