//
//  YHPlanScoreChart.swift
//  galaxy
//
//  Created by Dufet on 2025/3/25.
//  Copyright © 2025 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

class YHPlanScoreDrawView: UIView {
    
    let squareWidth = 180.0

    var points: [CGPoint] = []
    
    // 定义四个坐标点
    var scores: [CGFloat] = [] {
        didSet {
            
            guard scores.count == 4 else { return } // 确保有四个点

            let point1 = CGPoint(x: squareWidth/2.0, y: (1.0 - scores[0]/5.0)*squareWidth/2.0)
            let point2 = CGPoint(x: (1.0 + scores[1]/5.0)*squareWidth/2.0, y: squareWidth/2.0)
            let point3 = CGPoint(x: squareWidth/2.0, y: (1.0 + scores[2]/5.0)*squareWidth/2.0)
            let point4 = CGPoint(x: (1.0 - scores[3]/5.0)*squareWidth/2.0, y: squareWidth/2.0)
            self.points = [point1, point2, point3, point4]
            
            setNeedsDisplay() // 坐标变化时触发重绘
            
            if self.pointViews.count == points.count {
                for (index, view) in self.pointViews.enumerated() {
                    view.removeFromSuperview()
                    
                    // 微调点的位置
                    var point = self.points[index]
                    var gap = 1.0
                    if index == 0 || index == 3 {
                        gap = 1.0
                    } else {
                        gap = -1.0
                    }
                    if !isInteger(scores[index]) {
                        if index % 2 == 0 { // 垂直的两个点
                            point.y += gap
                        } else { // 水平的两个点
                            point.x += gap
                        }
                    }
                    
                    view.center = point
                    self.addSubview(view)
                }
            }
        }
    }
    
    func isInteger(_ value: CGFloat) -> Bool {
        return value.truncatingRemainder(dividingBy: 1) == 0
    }
    
    lazy var pointViews: [YHPlanScorePointView] = {
        let v1 = YHPlanScorePointView(frame: CGRect(x: 0, y: 0, width: 8, height: 8))
        let v2 = YHPlanScorePointView(frame: CGRect(x: 0, y: 0, width: 8, height: 8))
        let v3 = YHPlanScorePointView(frame: CGRect(x: 0, y: 0, width: 8, height: 8))
        let v4 = YHPlanScorePointView(frame: CGRect(x: 0, y: 0, width: 8, height: 8))
        return [v1, v2, v3, v4]
    }()

    override func draw(_ rect: CGRect) {
        guard points.count == 4 else { return } // 确保有四个点

        // 创建一个路径
        let path = UIBezierPath()

        // 移动到第一个点
        path.move(to: points[0])
        path.addLine(to: points[1])
        path.addLine(to: points[2])
        path.addLine(to: points[3])
        path.close()
        
        // 设置填充颜色为黑色，透明度50%
        UIColor.init(hex: 0xAE6C32).withAlphaComponent(0.1).setFill()
        path.fill()
        
        // 设置线条颜色为红色
        UIColor.init(hex: 0xAE6C32).setStroke()
        path.lineWidth = 1.0
        path.stroke()
    }
}

class YHPlanScoreItemView: UIView {
    
    lazy var titleLabel: UILabel = {
        let lable = UILabel()
        lable.textColor = UIColor.init(hex: 0x604127)
        lable.textAlignment = .center
        lable.font = UIFont.PFSC_B(ofSize: 14)
        return lable
    }()
    
    lazy var scoreLabel: UILabel = {
        let lable = UILabel()
        lable.textColor = UIColor.init(hex: 0xAE6C32)
        lable.textAlignment = .center
        lable.font = UIFont(name: "DINAlternate-Bold", size: 16)
        return lable
    }()
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        createUI()
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    func createUI() {
        self.addSubview(titleLabel)
        self.addSubview(scoreLabel)
        
        titleLabel.snp.makeConstraints { make in
            make.centerX.equalToSuperview()
            make.height.equalTo(20)
        }
        
        scoreLabel.snp.makeConstraints { make in
            make.centerX.equalToSuperview()
            make.height.equalTo(16)
            make.top.equalTo(titleLabel.snp.bottom).offset(2)
        }
    }
}

class YHPlanScorePointView: UIView {
    
    lazy var bigCircleView: UIView = {
        let v = UIView()
        v.backgroundColor = UIColor.init(hex: 0xAE6C32, alpha: 0.2)
        v.layer.cornerRadius = 4.0
        return v
    }()
    
    lazy var smallCircleView: UIView = {
        let v = UIView()
        v.backgroundColor = UIColor.init(hex: 0xAE6C32)
        v.layer.cornerRadius = 2.0
        return v
    }()
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        createUI()
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    func createUI() {
        
        self.addSubview(bigCircleView)
        self.addSubview(smallCircleView)
        
        bigCircleView.snp.makeConstraints { make in
            make.width.height.equalTo(8)
            make.centerX.equalToSuperview()
            make.centerY.equalToSuperview()
        }
        smallCircleView.snp.makeConstraints { make in
            make.width.height.equalTo(4)
            make.centerX.equalToSuperview()
            make.centerY.equalToSuperview()
        }
    }
}

class YHPlanScoreChart: UIView {
    
    let squareWidth = 180.0
    
    lazy var squareImgV: UIImageView = {
        let v = UIImageView()
        v.image = UIImage(named: "plan_score_square")
        return v
    }()
    
    lazy var circleBgImgV: UIImageView = {
        let v = UIImageView()
        v.image = UIImage(named: "plan_score_circle_bg")
        return v
    }()
    
    lazy var drawView: YHPlanScoreDrawView = {
        let v = YHPlanScoreDrawView(frame: .zero)
        v.backgroundColor = .clear
        return v
    }()
    
    lazy var careerItemView: YHPlanScoreItemView = {
        let v = YHPlanScoreItemView(frame: .zero)
        v.titleLabel.text = "工作"
        v.scoreLabel.text = "0"
        return v
    }()
    
    lazy var lifeItemView: YHPlanScoreItemView = {
        let v = YHPlanScoreItemView(frame: .zero)
        v.titleLabel.text = "生活"
        v.scoreLabel.text = "0"
        return v
    }()
    
    lazy var investItemView: YHPlanScoreItemView = {
        let v = YHPlanScoreItemView(frame: .zero)
        v.titleLabel.text = "投资"
        v.scoreLabel.text = "0"
        return v
    }()
    
    lazy var stayItemView: YHPlanScoreItemView = {
        let v = YHPlanScoreItemView(frame: .zero)
        v.titleLabel.text = "逗留"
        v.scoreLabel.text = "0"
        return v
    }()

    override init(frame: CGRect) {
        super.init(frame: frame)
        createUI()
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    func updateScores(_ scores: [CGFloat]) {
        
        guard scores.count == 4 else { return } // 确保有四个点
        
        self.drawView.scores = scores
        
        careerItemView.scoreLabel.text = "\(scores[0])"
        lifeItemView.scoreLabel.text = "\(scores[1])"
        investItemView.scoreLabel.text = "\(scores[2])"
        stayItemView.scoreLabel.text = "\(scores[3])"
    }
    
    func createUI() {
        self.addSubview(circleBgImgV)
        self.addSubview(squareImgV)
        self.addSubview(careerItemView)
        self.addSubview(lifeItemView)
        self.addSubview(investItemView)
        self.addSubview(stayItemView)
        squareImgV.addSubview(drawView)

        squareImgV.snp.makeConstraints { make in
            make.height.width.equalTo(squareWidth)
            make.centerX.equalToSuperview()
            make.centerY.equalToSuperview()
        }
        
        circleBgImgV.snp.makeConstraints { make in
            make.height.width.equalTo(228)
            make.centerX.equalToSuperview()
            make.centerY.equalToSuperview()
        }
        
        drawView.snp.makeConstraints { make in
            make.edges.equalToSuperview()
        }
        
        careerItemView.snp.makeConstraints { make in
            make.centerX.equalToSuperview()
            make.bottom.equalTo(squareImgV.snp.top).offset(-5)
            make.height.equalTo(38)
            make.width.equalTo(40)
        }
        
        lifeItemView.snp.makeConstraints { make in
            make.centerY.equalToSuperview()
            make.left.equalTo(squareImgV.snp.right).offset(5)
            make.height.equalTo(38)
            make.width.equalTo(40)
        }
        
        investItemView.snp.makeConstraints { make in
            make.centerX.equalToSuperview()
            make.top.equalTo(squareImgV.snp.bottom).offset(5)
            make.height.equalTo(38)
            make.width.equalTo(40)
        }
        
        stayItemView.snp.makeConstraints { make in
            make.centerY.equalToSuperview()
            make.right.equalTo(squareImgV.snp.left).offset(-5)
            make.height.equalTo(38)
            make.width.equalTo(40)
        }
        
    }
}
