//
//  YHScanViewController.swift
//  galaxy
//
//  Created by alexzzw on 2025/9/18.
//  Copyright © 2025 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

// MARK: - 扫码控制器
class YHScanViewController: LBXScanViewController {
    
    // MARK: - Properties
    private let topOffset = 44.0
    private let leftOffset = 48.0
    
    /// 扫码支持的类型(可以是单个或多个类型的组合)
    var scanTypes: YHScanType? {
        didSet {
            updateUIForScanType()
        }
    }
    
    /// 扫码成功回调
    var scanCompletion: YHScanCompletion?
    
    /// 提示文字标签
    private lazy var instructionLabel: UILabel = {
        let label = UILabel()
        label.textColor = UIColor(hexString: "#EBF0F9")
        label.font = .PFSC_M(ofSize: 18)
        label.textAlignment = .center
        label.numberOfLines = 0
        return label
    }()
    
    // MARK: - Lifecycle
    override var preferredStatusBarStyle: UIStatusBarStyle {
        return .lightContent
    }

    override func viewDidLoad() {
        super.viewDidLoad()
        setupNavigationBar()
        setupScanStyle()
        setupInstructionLabel()
        updateUIForScanType()
    }
    
    override func viewDidAppear(_ animated: Bool) {
        super.viewDidAppear(animated)
        let navBar = gk_navigationBar
        if view.subviews.contains(navBar) {
            view.bringSubviewToFront(navBar)
        }
        if view.subviews.contains(instructionLabel) {
            view.bringSubviewToFront(instructionLabel)
        }
    }
    
    override func handleCodeResult(arrayResult: [LBXScanResult]) {
        guard let firstResult = arrayResult.first,
              let code = firstResult.strScanned else {
            return
        }
        guard let scanTypes = scanTypes else {
            super.handleCodeResult(arrayResult: arrayResult)
            return
        }
        // 停止扫描
        stopScan()
        
        // 识别二维码类型
        let recognizedType = scanTypes.recognizeType(from: code)
        
        // 验证类型
        if recognizedType == nil && scanTypes != .all && scanTypes.rawValue != 0 {
            // 类型不匹配,显示错误提示
            showError(message: "二维码类型不匹配,请扫描正确的二维码")
            // 延迟后重新开始扫描
            DispatchQueue.main.asyncAfter(deadline: .now() + 1.5) { [weak self] in
                self?.startScan()
            }
            return
        }
        
        // 震动反馈
        let generator = UINotificationFeedbackGenerator()
        generator.notificationOccurred(.success)
        
        // 创建结果
        let result = YHScanResult(
            code: code,
            recognizedType: recognizedType,
            supportedTypes: scanTypes
        )
        
        // 回调处理
        if let completion = scanCompletion {
            navigationController?.popViewController(animated: true)
            completion(result)
            
        } else {
            // 如果没有设置回调,使用默认处理
            handleDefaultResult(result)
        }
    }
    
    deinit {
        print("YHScanViewController deinit")
    }
}

// MARK: - Setup
extension YHScanViewController {
    
    private func setupNavigationBar() {
        gk_navTitleColor = .white
        gk_navBarAlpha = 0.0
        gk_navBackgroundColor = .clear
        gk_backImage = UIImage(named: "back_icon_white")
    }
    
    private func setupScanStyle() {
        var style = LBXScanViewStyle()
        style.centerUpOffset = topOffset
        style.xScanRetangleOffset = leftOffset
        style.photoframeAngleStyle = LBXScanViewPhotoframeAngleStyle.Inner
        style.photoframeLineW = 3
        style.photoframeAngleW = 18
        style.photoframeAngleH = 18
        style.isNeedShowRetangle = false
        style.anmiationStyle = LBXScanViewAnimationStyle.LineMove
        style.animationImage = UIImage(named: "scan_move_line")
        style.colorAngle = UIColor.white
        style.color_NotRecoginitonArea = UIColor.black.withAlphaComponent(0.3)
        scanStyle = style
    }
    
    private func setupInstructionLabel() {
        view.addSubview(instructionLabel)
        instructionLabel.snp.makeConstraints { make in
            make.centerX.equalToSuperview()
            make.bottom.equalTo(view.safeAreaLayoutGuide.snp.bottom).offset(-45)
            make.left.equalToSuperview().offset(20)
            make.right.equalToSuperview().offset(-20)
        }
    }
    
    private func updateUIForScanType() {
        guard let scanTypes = scanTypes else {
            gk_navTitle = YHScanType.all.title
            instructionLabel.text = YHScanType.all.instructionText
            return
        }
        gk_navTitle = scanTypes.title
        instructionLabel.text = scanTypes.instructionText
    }
}

// MARK: - Result Handling
extension YHScanViewController {
    
    /// 默认结果处理(当没有设置回调时)
    private func handleDefaultResult(_ result: YHScanResult) {
        guard let type = result.recognizedType else {
            showSuccessAndPop(message: "扫码成功")
            return
        }
        
        if type == .checkIn {
            handleCheckIn(code: result.code)
        } else if type == .smartCabinet {
            handleSmartCabinet(code: result.code)
        }
    }
    
    // MARK: - 业务处理方法
    
    private func handleCheckIn(code: String) {
        printLog("签到扫码: \(code)")
        // TODO: 调用签到 API
        showSuccessAndPop(message: "签到成功")
    }
    
    private func handleSmartCabinet(code: String) {
        printLog("智能柜扫码: \(code)")
        // TODO: 调用智能柜开柜 API
        showSuccessAndPop(message: "开柜成功")
    }
    
    // MARK: - Helper
    
    private func showError(message: String) {        
        YHCommonAlertView.show("提示", message, "", "确定", fullGuestureEnable: false) {
        } callBack: {
        }
    }
    
    private func showSuccessAndPop(message: String) {
        YHCommonAlertView.show("成功", message, "", "确定", fullGuestureEnable: false) {
        } callBack: { [weak self] in
            self?.navigationController?.popViewController(animated: true)
        }
    }
}

// MARK: - 便捷初始化
extension YHScanViewController {
    
    /// 便捷初始化方法
    /// - Parameters:
    ///   - types: 扫码支持的类型(可以是单个或多个的组合)
    ///   - completion: 扫码成功回调
    /// - Returns: 扫码控制器实例
    static func create(types: YHScanType, completion: YHScanCompletion? = nil) -> YHScanViewController {
        let vc = YHScanViewController()
        vc.scanTypes = types
        vc.scanCompletion = completion
        return vc
    }
}

// MARK: - 使用示例
/*
 
 // 示例1: 单一类型 - 仅签到
 let checkInVC = YHScanViewController.create(types: .checkIn) { result in
 print("签到码: \(result.code)")
 print("识别类型: \(result.recognizedType)")
 }
 navigationController?.pushViewController(checkInVC, animated: true)
 
 // 示例2: 单一类型 - 仅智能柜
 let cabinetVC = YHScanViewController.create(types: .smartCabinet) { result in
 print("智能柜码: \(result.code)")
 }
 navigationController?.pushViewController(cabinetVC, animated: true)
 
 // 示例3: 复合类型 - 同时支持签到和智能柜 ⭐
 let multiVC = YHScanViewController.create(types: [.checkIn, .smartCabinet]) { result in
 print("扫码内容: \(result.code)")
 
 // 根据识别出的类型进行不同处理
 if let type = result.recognizedType {
 if type == .checkIn {
 print("这是签到码")
 APIManager.checkIn(code: result.code)
 } else if type == .smartCabinet {
 print("这是智能柜码")
 APIManager.openCabinet(code: result.code)
 }
 }
 }
 navigationController?.pushViewController(multiVC, animated: true)
 
 // 示例4: 所有类型
 let allTypesVC = YHScanViewController.create(types: .all) { result in
 // 自动识别并处理
 }
 navigationController?.pushViewController(allTypesVC, animated: true)
 
 // 示例5: 二维码格式示例
  // 签到码格式: "CHECKIN_xxxxx"
  // 智能柜码格式: "CABINET_xxxxx"
  // 或 JSON 格式: {"type": "checkin", "data": "xxxxx"}
  
  // 示例6: 后续扩展新类型
  /*
  struct YHScanType: OptionSet {
      let rawValue: Int
      
      static let checkIn       = YHScanType(rawValue: 1 << 0)
      static let smartCabinet  = YHScanType(rawValue: 1 << 1)
      static let document      = YHScanType(rawValue: 1 << 2)  // ← 新增
      static let payment       = YHScanType(rawValue: 1 << 3)  // ← 新增
  }
  
  // 使用组合
  let vc = YHScanViewController.create(types: [.checkIn, .document]) { result in
      // 同时支持签到和文档扫码
  }
  */
 
 */
