//
//  LBXScanViewController.swift
//  swiftScan
//
//  Created by lbxia on 15/12/8.
//  Copyright © 2015年 xialibing. All rights reserved.
//

import UIKit
import Foundation
import AVFoundation

protocol LBXScanViewControllerDelegate: AnyObject {
     func scanFinished(scanResult: LBXScanResult?, error: String?)
}

protocol QRRectDelegate: AnyObject {
    func drawwed()
}

class LBXScanViewController: YHBaseViewController, UIImagePickerControllerDelegate, UINavigationControllerDelegate {
    // 返回扫码结果，也可以通过继承本控制器，改写该handleCodeResult方法即可
    open weak var scanResultDelegate: LBXScanViewControllerDelegate?

    open var delegate: QRRectDelegate?

    open var scanObj: LBXScanWrapper?

    open var scanStyle: LBXScanViewStyle? = LBXScanViewStyle()

    open var qRScanView: LBXScanView?

    // 启动区域识别功能
    open var isOpenInterestRect = false

    // 识别码的类型
    public var arrayCodeType: [AVMetadataObject.ObjectType]?

    // 是否需要识别后的当前图像
    public var isNeedCodeImage = false

    // 相机启动提示文字
    public var readyString: String = "loading"

    override open func viewDidLoad() {
        super.viewDidLoad()
        self.view.backgroundColor = UIColor.black
//        self.edgesForExtendedLayout = UIRectEdge(rawValue: 0)
    }

    open func setNeedCodeImage(needCodeImg: Bool) {
        isNeedCodeImage = needCodeImg
    }

    // 设置框内识别
    open func setOpenInterestRect(isOpen: Bool) {
        isOpenInterestRect = isOpen
    }

    override open func viewDidAppear(_ animated: Bool) {
        super.viewDidAppear(animated)
        drawScanView()

        LBXPermissions.authorizeCameraWith { [weak self] granted in

            if granted {
                self?.perform(#selector(LBXScanViewController.startScan), with: nil, afterDelay: 0.3)
            } else {
                // 跳转到系统设置
                YHCommonAlertView.show("", "没有相机权限，是否跳转设置页面开启？", "取消", "确定", fullGuestureEnable: false) {
                    
                } callBack: {
                    let localUrl = URL(string: UIApplication.openSettingsURLString)
                    if let url = localUrl {
                        if UIApplication.shared.canOpenURL(url) {
                            UIApplication.shared.open(url, options: [:], completionHandler: nil)
                        }
                    }
                }
            }
        }
    }
    
    override open func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        NSObject.cancelPreviousPerformRequests(withTarget: self)
        stopScan()
    }

    @objc open func startScan() {
        if scanObj == nil {
            var cropRect = CGRect.zero
            if isOpenInterestRect {
                guard let style = scanStyle else { return }
                cropRect = LBXScanView.getScanRectWithPreView(preView: self.view, style: style)
            }

            // 指定识别几种码
            if arrayCodeType == nil {
                arrayCodeType = [.qr, .ean13, .code39, .code128]
            }
            
            guard let codeTypes = arrayCodeType else { return }
            
            scanObj = LBXScanWrapper(videoPreView: self.view, objType: codeTypes, isCaptureImg: isNeedCodeImage, cropRect: cropRect, success: { [weak self] arrayResult in
                if let strongSelf = self {
                    // 停止扫描动画
                    strongSelf.qRScanView?.stopScanAnimation()
                    strongSelf.handleCodeResult(arrayResult: arrayResult)
                }
            })
        }

        // 结束相机等待提示
        qRScanView?.deviceStopReadying()

        // 开始扫描动画
        qRScanView?.startScanAnimation()

        // 相机运行
        scanObj?.start()
    }

    open func stopScan() {
        qRScanView?.stopScanAnimation()
        scanObj?.stop()
    }

    open func drawScanView() {
        if qRScanView == nil {
            guard let style = scanStyle else { return }
            let qRScanView = LBXScanView(frame: self.view.frame, vstyle: style)
            self.qRScanView = qRScanView
            self.view.addSubview(qRScanView)
            delegate?.drawwed()
        }
        qRScanView?.deviceStartReadying(readyStr: readyString)
    }

    // 处理扫码结果，如果是继承本控制器的，可以重写该方法,作出相应地处理，或者设置delegate作出相应处理
    open func handleCodeResult(arrayResult: [LBXScanResult]) {
        if let delegate = scanResultDelegate {
            self.navigationController?.popViewController(animated: true)
            if arrayResult.count > 0 {
                let result: LBXScanResult = arrayResult[0]
                delegate.scanFinished(scanResult: result, error: nil)
            } else {
                delegate.scanFinished(scanResult: nil, error: "扫码结束，无结果")
            }
            
        } else {
            for result: LBXScanResult in arrayResult {
                debugPrint("%@", result.strScanned ?? "")
            }
            if arrayResult.count > 0 {
                let result: LBXScanResult = arrayResult[0]
                YHCommonAlertView.show(result.strBarCodeType, result.strScanned, "", "确定", fullGuestureEnable: false) {
                } callBack: { [weak self] in
                    self?.startScan()
                }
            } else {
                YHCommonAlertView.show("", "扫码结束，无结果", "", "确定", fullGuestureEnable: false) {
                } callBack: { [weak self] in
                    self?.startScan()
                }
            }
        }
    }

    open func openPhotoAlbum() {
        LBXPermissions.authorizePhotoWith { [weak self] granted in
            if granted {
                let picker = UIImagePickerController()
                picker.sourceType = .photoLibrary
                picker.delegate = self
                picker.allowsEditing = true
                self?.present(picker, animated: true, completion: nil)
            } else {
                YHCommonAlertView.show("", "没有相册权限，是否跳转设置页面开启？", "取消", "确定", fullGuestureEnable: false) {
                    
                } callBack: {
                    let localUrl = URL(string: UIApplication.openSettingsURLString)
                    if let url = localUrl {
                        if UIApplication.shared.canOpenURL(url) {
                            UIApplication.shared.open(url, options: [:], completionHandler: nil)
                        }
                    }
                }
            }
        }
    }

    // MARK: - 相册选择图片识别二维码 （条形码没有找到系统方法）

    public func imagePickerController(_ picker: UIImagePickerController, didFinishPickingMediaWithInfo info: [UIImagePickerController.InfoKey: Any]) {
        picker.dismiss(animated: true, completion: nil)
        var image: UIImage? = info[.editedImage] as? UIImage
        if image == nil {
            image = info[.originalImage] as? UIImage
        }
        if let validImage = image {
            let arrayResult = LBXScanWrapper.recognizeQRImage(image: validImage)
            if arrayResult.count > 0 {
                handleCodeResult(arrayResult: arrayResult)
                return
            }
        }
        YHCommonAlertView.show("", "暂无法识别图片中二维码", "", "确定", fullGuestureEnable: false) {
        } callBack: {
        }
    }

    deinit {
        
    }
}
