//
//  YHFileExplainReasonInputView.swift
//  galaxy
//
//  Created by EDY on 2024/10/10.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit
import AttributedString

class YHFileExplainReasonInputView: UIView {
    static let height = 294.0
    var maxWordsCount = 50
    var editBlock: ((String?) -> Void)?
    var dismissBlock: (() -> Void)?

    lazy var blackMaskView: UIView = {
        let view = UIView(frame: UIScreen.main.bounds)
        view.backgroundColor = UIColor(hex: 0x0F1214, alpha: 0.5)
        let tap = UITapGestureRecognizer(target: self, action: #selector(dismiss))
        view.addGestureRecognizer(tap)
        return view
    }()
    
    lazy var whiteContentView: UIView = {
        let view = UIView()
        view.backgroundColor = .white
        let tap = UITapGestureRecognizer(target: self, action: #selector(didWhiteContentViewTap))
        view.addGestureRecognizer(tap)
        return view
    }()
    
    lazy var titleLabel: UILabel = {
        let label = UILabel()
        label.textColor = UIColor.mainTextColor
        label.textAlignment = .center
        label.font = UIFont.PFSC_B(ofSize: 17)
        label.text = "补充说明"
        return label
    }()
    
    lazy var alertLabel: UILabel = {
        let label = UILabel()
        let a = ASAttributedString.init("*", .font(UIFont.PFSC_R(ofSize: 14)), .foreground(UIColor.failColor))
        let b = ASAttributedString.init("若无法提供材料，请填写原因：", .font(UIFont.PFSC_R(ofSize: 14)), .foreground(UIColor.mainTextColor))
        label.attributed.text = a + b
        return label
    }()
    
    lazy var countLabel: UILabel = {
        let label = UILabel()
        label.textColor = UIColor.mainTextColor18
        label.textAlignment = .right
        label.font = UIFont.PFSC_B(ofSize: 16)
        label.text = "0/0"
        return label
    }()
    
    lazy var reasonLabel: UILabel = {
        let label = UILabel()
        label.textColor = UIColor.failColor
        label.font = UIFont.PFSC_B(ofSize: 12)
        label.text = "请填写原因"
        label.isHidden = true
        return label
    }()
    
    lazy var closeBtn: UIButton = {
        let btn = UIButton()
        btn.setImage(UIImage(named: "my_cer_sheet_close"), for: .normal)
        btn.addTarget(self, action: #selector(dismiss), for: .touchUpInside)
        return btn
    }()
    
    lazy var topLine: UIView = {
       let view = UIView()
       view.backgroundColor = .separatorColor
       return view
    }()
    
    lazy var inputContentView: UIView = {
        let view = UIView()
        view.backgroundColor = .contentBkgColor
        view.layer.cornerRadius = kCornerRadius3
        view.clipsToBounds = true
        return view
    }()
    
    lazy var textField: UITextView = {
        let textField = UITextView()
        textField.backgroundColor = .clear
        textField.font = UIFont.PFSC_M(ofSize: 16)
        textField.tintColor = UIColor.brandMainColor
        textField.textColor = UIColor.mainTextColor
        textField.delegate = self
        textField.returnKeyType = .done
        return textField
    }()
    
    lazy var cancelBtn: UIButton = {
        let btn = UIButton()
        btn.setTitle("取消", for: .normal)
        btn.titleLabel?.font = UIFont.PFSC_M(ofSize: 14)
        btn.setTitleColor(UIColor.mainTextColor, for: .normal)
        btn.backgroundColor = .contentBkgColor
        btn.layer.cornerRadius = kCornerRadius3
        btn.addTarget(self, action: #selector(didClickCancelBtn), for: .touchUpInside)
        return btn
    }()
    
    lazy var confirmBtn: UIButton = {
        let btn = UIButton()
        btn.setTitle("确定".local, for: .normal)
        btn.titleLabel?.font = UIFont.PFSC_M(ofSize: 14)
        btn.setTitleColor(.white, for: .normal)
        btn.backgroundColor = UIColor.brandMainColor
        btn.layer.cornerRadius = kCornerRadius3
        btn.addTarget(self, action: #selector(didClickConfirmBtn), for: .touchUpInside)
        return btn
    }()
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        createUI()
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    static func inputView(defalutText: String?, _  placeholde: String = "请输入文件名称", _ title: String = "补充说明") -> YHFileExplainReasonInputView {
        let view = YHFileExplainReasonInputView(frame: CGRect(x: 0, y: 0, width: KScreenWidth, height: Self.height))
        view.textField.text = defalutText
        view.titleLabel.text = title
        return view
    }
    
    func createUI() {

        self.addSubview(whiteContentView)
        whiteContentView.addSubview(titleLabel)
        whiteContentView.addSubview(closeBtn)
        whiteContentView.addSubview(topLine)
        whiteContentView.addSubview(alertLabel)
        whiteContentView.addSubview(inputContentView)
        inputContentView.addSubview(textField)
        inputContentView.addSubview(countLabel)
        whiteContentView.addSubview(cancelBtn)
        whiteContentView.addSubview(confirmBtn)
        whiteContentView.addSubview(reasonLabel)
        
        let corner = UIRectCorner(rawValue: UIRectCorner.topLeft.rawValue | UIRectCorner.topRight.rawValue)
        let path = UIBezierPath(roundedRect: bounds, byRoundingCorners: corner, cornerRadii: CGSize(width: kCornerRadius6, height: kCornerRadius6))
        let layer = CAShapeLayer()
        layer.frame = whiteContentView.bounds
        layer.path = path.cgPath
        whiteContentView.layer.mask = layer
        
        whiteContentView.snp.makeConstraints { make in
            make.left.right.top.bottom.equalToSuperview()
        }
        titleLabel.snp.makeConstraints { make in
            make.left.right.top.equalToSuperview()
            make.height.equalTo(52.0)
        }
        closeBtn.snp.makeConstraints { make in
            make.width.height.equalTo(24.0)
            make.right.equalToSuperview().offset(-20)
            make.centerY.equalTo(titleLabel)
        }
        topLine.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(20)
            make.right.equalToSuperview().offset(-20)
            make.top.equalTo(titleLabel.snp.bottom)
            make.height.equalTo(1)
        }
        alertLabel.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(20)
            make.right.equalToSuperview().offset(-20)
            make.top.equalTo(72)
            make.height.equalTo(20)
        }
        inputContentView.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(20)
            make.right.equalToSuperview().offset(-20)
            make.top.equalTo(108)
            make.height.equalTo(88)
        }
        textField.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(16)
            make.right.equalToSuperview().offset(-16)
            make.height.equalTo(88)
            make.top.equalToSuperview()
        }
        countLabel.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(16)
            make.right.equalToSuperview().offset(-16)
            make.height.equalTo(20)
            make.bottom.equalTo(-14)
        }
        reasonLabel.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(16)
            make.right.equalToSuperview().offset(-16)
            make.height.equalTo(20)
            make.top.equalTo(inputContentView.snp.bottom).offset(4)
        }
        cancelBtn.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(16)
            make.right.equalTo(whiteContentView.snp.centerX).offset(-6)
            make.top.equalTo(inputContentView.snp.bottom).offset(32)
            make.height.equalTo(45)
        }
        confirmBtn.snp.makeConstraints { make in
            make.right.equalToSuperview().offset(-16)
            make.left.equalTo(whiteContentView.snp.centerX).offset(6)
            make.top.equalTo(cancelBtn)
            make.height.equalTo(45)
        }
    }
}

extension YHFileExplainReasonInputView {
    
    @objc func dismiss() {
        self.textField.resignFirstResponder()
        UIApplication.shared.yhKeyWindow()?.endEditing(true)
        self.dismissBlock?()
    }
    
    @objc func didWhiteContentViewTap() {
        self.textField.becomeFirstResponder()
    }
    
    @objc func didClickCancelBtn() {
        dismiss()
    }
    
    @objc func didClickConfirmBtn() {
        if isEmptyString(textField.text) {
            reasonLabel.isHidden = false
            return
        }
        if let editBlock = editBlock {
            editBlock(textField.text)
        }
        dismiss()
    }
}

extension YHFileExplainReasonInputView: UITextViewDelegate {
    func textViewDidChange(_ textView: UITextView) {
        let maxLength = 50
        let currentString: String = textView.text
        
        if currentString.count >= maxLength {
            // 截取字符串至最大长度
            textView.text = String(currentString.prefix(maxLength))
        }
        countLabel.text = "\(textView.text.count)/50"
    }
}
