//
//  YHCertificateSearchBar.swift
//  galaxy
//
//  Created by edy on 2024/3/13.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

class YHCertificateSearchBar: UIView {

    static let height = 40.0
    static let cancelBtnHeight = height-8.0
    static let maxWordsCount = 50
    
    var textChange:((String?)->Void)?
    var searchBlock:((String?)->Void)?
    var placeHolder:String? {
        didSet {
            if let placeholder = placeHolder {
                textField.attributedPlaceholder = NSAttributedString(string: placeholder, attributes: [NSAttributedString.Key.foregroundColor : UIColor.placeHolderColor])
            }
        }
    }
    private lazy var contentView = {
        let view = UIView()
        view.backgroundColor = .contentBkgColor
        view.layer.cornerRadius = Self.height/2.0
        view.clipsToBounds = true
        return view
    }()
    
    lazy var textField:UITextField = {
        let textField = UITextField()
        textField.backgroundColor = .clear
        textField.attributedPlaceholder = NSAttributedString(string: "请输入关键词搜索".local, attributes: [NSAttributedString.Key.foregroundColor : UIColor.placeHolderColor])
        textField.font = UIFont.PFSC_M(ofSize: 12)
        textField.tintColor = UIColor.brandMainColor
        textField.textColor = UIColor.mainTextColor
        textField.clearButtonMode = .whileEditing
        textField.addTarget(self, action: #selector(textFieldChanged(textField:)), for: .editingChanged)
        textField.delegate = self
        return textField
    }()
    
    lazy var searchBtn: UIButton = {
        let btn = UIButton()
        btn.setTitle("搜索".local, for: .normal)
        btn.titleLabel?.font = UIFont.PFSC_M(ofSize: 12)
        btn.setTitleColor(.white, for: .normal)
        btn.backgroundColor = UIColor.brandMainColor
        btn.layer.cornerRadius = Self.cancelBtnHeight/2.0
        btn.addTarget(self, action: #selector(searchBtnClicked), for: .touchUpInside)
        return btn
    }()
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        self.createUI()
    }
    
    required init?(coder: NSCoder) {
        super.init(coder: coder)
    }
    
    static func createBar() -> YHCertificateSearchBar {
        let view = YHCertificateSearchBar(frame: CGRect(x: 0, y: 0, width: KScreenWidth-32, height: height))
        return view
    }
    
    @objc func searchBtnClicked() {
        if let searchBlock = searchBlock {
            searchBlock(textField.text)
        }
    }
    
    func setSearchButtonEnable(_ enable: Bool) {
        searchBtn.isUserInteractionEnabled = enable
        searchBtn.alpha = (enable ? 1.0 : 0.5)
    }
    
    func createUI() {
        
        self.backgroundColor = .white
        setSearchButtonEnable(false)

        contentView.addSubview(textField)
        contentView.addSubview(searchBtn)
        self.addSubview(contentView)
        
        contentView.snp.makeConstraints { make in
            make.edges.equalToSuperview()
        }
        
        textField.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(Self.height/2.0)
            make.right.equalTo(searchBtn.snp.left).offset(-4)
            make.centerY.equalToSuperview()
        }
        
        searchBtn.snp.makeConstraints { make in
            make.size.equalTo(CGSize(width: 66, height: Self.height-8.0))
            make.centerY.equalToSuperview()
            make.right.equalToSuperview().offset(-4)
        }
    }
    
    @objc func textFieldChanged(textField:UITextField) {
        if isEmptyString(textField.text)  {
            setSearchButtonEnable(false)
        } else {
            setSearchButtonEnable(true)
        }
        if let textChange = textChange {
            textChange(textField.text)
        }
    }
}

extension YHCertificateSearchBar: UITextFieldDelegate {
    
    func textField(_ textField: UITextField, shouldChangeCharactersIn range: NSRange, replacementString string: String) -> Bool {
        
        if string.isEmpty == true { // 点击删除
            return true
        }
        
        var newText = (textField.text! as NSString).replacingCharacters(in: range, with: string)
        if newText.count > Self.maxWordsCount {
            YHHUD.flash(message: "搜索限制最多\(Self.maxWordsCount)个字符")
            return false
        }
        return true
    }
}
