//
//  YHButlerServiceManager.swift
//  galaxy
//
//  Created by alexzzw on 2024/9/4.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit
import IQKeyboardManagerSwift
import SafariServices

class YHButlerServiceManager: NSObject {
    
    static let shared = YHButlerServiceManager()
    
    private weak var conversationManager = QYSDK.shared().conversationManager()
    
    // 调整UI
    private weak var customUIConfig = QYSDK.shared().customUIConfig()
    
    // 点击事件处理
    private weak var customActionConfig = QYSDK.shared().customActionConfig()
    
    private lazy var getUserInfoViewModel: YHButlerServiceViewModel = YHButlerServiceViewModel()
    
    private var curentPresentedVC: UIViewController?
    
    private lazy var reportHandler: YHButlerMessageReportHandler = YHButlerMessageReportHandler()
    private var groupTmpId: Int64?
    
    var lastMessage: YHButlerServiceMessage? {
        getLastMessage()
    }
    
    var unreadCount: Int {
        allUnreadCount()
    }
    
    var guideWindowShowedKey: String? {
        guard let userId = YHLoginManager.shared.userModel?.id, userId.count > 0 else {
            return nil
        }
        return "guideWindowShowedKey" + "_\(userId)"
    }
    
    var didShowWindowDict: [String: Bool] = [:]
        
    private override init() {
        super.init()
        conversationManager?.setDelegate(self)
        
        NotificationCenter.default.addObserver(self, selector: #selector(loginSuccess), name: YhConstant.YhNotification.didLoginSuccessNotifiction, object: nil)
        NotificationCenter.default.addObserver(self, selector: #selector(logoutSuccess), name: YhConstant.YhNotification.didLogoutSuccessNotifiction, object: nil)
//        NotificationCenter.default.addObserver(self, selector: #selector(didRevUatParam), name: YhConstant.YhNotification.didRevUatParam, object: nil)
    }
    
    deinit {
        NotificationCenter.default.removeObserver(self)
    }
        
    func setupSDK() {
        let qyOption = QYSDKOption(appKey: YhConstant.QiYuSDK.appKey)
        qyOption.appName = YhConstant.QiYuSDK.appName
        QYSDK.shared().register(with: qyOption)
        updateActionConfig()
        /*
        QYSDK.shared().registerPushMessageNotification { pushMessage in
            os_log("#####registerPushMessageNotification %@ %ld %f", (pushMessage?.text ?? ""), (pushMessage?.type.rawValue ?? 0), (pushMessage?.time ?? 0))
            printLog("$$$$registerPushMessageNotification \(pushMessage?.text ?? "") \(pushMessage?.type.rawValue ?? 0) \(pushMessage?.time ?? 0)")
        }
        */
        guard isUatAllowed() else {
            return
        }
        initSDKAccountAndLastMessage { _ in
            //
        }
    }
            
    func updateApnsToken(token: Data) {
        QYSDK.shared().updateApnsToken(token)
    }
    
    @MainActor func gotoButlerServiceSessionVC(sourceTitle: String? = nil, sourceUrl: String? = nil, sessionTitle: String = "银河管家", navigationController: UINavigationController?, completion: @escaping (Bool) -> Void) {
        
        func gotoSessionVC(_ groupTmpId: Int64? = nil) {
            let source = QYSource()
            source.title = sourceTitle
            source.urlString = sourceUrl
            IQKeyboardManager.shared.disabledDistanceHandlingClasses.append(QYSessionViewController.self)
            guard let sessionViewController = QYSDK.shared().sessionViewController() else {
                completion(false)
                return
            }
            completion(true)
            if let groupTmpId = groupTmpId {
                sessionViewController.shuntTemplateId = groupTmpId
            }
            sessionViewController.sessionTitle = sessionTitle
            sessionViewController.source = source

            if isIOS17() {
                let nav = UINavigationController(rootViewController: sessionViewController)
                let backButton = UIButton(type: .custom)
                backButton.setTitle("返回", for: .normal)
                backButton.setTitleColor(.brandMainColor, for: .normal)
                backButton.titleLabel?.font = .PFSC_R(ofSize: 15)
                backButton.frame = CGRect(x: 0, y: 0, width: 60, height: 30)
                backButton.addTarget(self, action: #selector(sessionGoBack), for: .touchUpInside)
                sessionViewController.navigationItem.leftBarButtonItem = UIBarButtonItem(customView: backButton)
                nav.modalPresentationStyle = .fullScreen
                self.curentPresentedVC = nav
                UIViewController.current?.present(nav, animated: true, completion: nil)
            } else {
                navigationController?.pushViewController(sessionViewController)
            }
        }
        
        guard let userId = YHLoginManager.shared.userModel?.id, userId.count > 0 else {
            completion(false)
            return
        }
        
        getUserInfoViewModel.getUserInfomation(userId) { [weak self] dataString, avatar, groupTmpId, _ in
            // 改变头像
            self?.customUIConfig?.customerHeadImageUrl = avatar
            self?.customUIConfig?.showServiceNickName = true
            self?.updateUserInfo(userId: userId, userData: dataString ?? "", completion: { _ in
                DispatchQueue.main.async {
                    gotoSessionVC(groupTmpId)
                }
            })
        }
    }
}

extension YHButlerServiceManager {
    // MARK: - customActionConfig
    
    private func handelMessageType(_ content: String?) -> QYLinkClickActionPolicy {
        
        if let dicData = content, let data = dicData.data(using: .utf8) {
            do {
                if let jsonObject = try JSONSerialization.jsonObject(with: data, options: []) as? [String: Any] {
                    // 转换成功，jsonObject 是一个字典
                    printLog("JSON字符串转换为字典成功: \(jsonObject)")
                    /*
                     "{\"order_id\":151824,\"type\":27}"
                     */
                    
                    if !jsonObject.isEmpty {
                        let msgType: String = jsonObject["msg_type"] as? String ?? ""
                        let navH5Url: String = jsonObject["nav_h5_url"] as? String ?? ""
                        let orderID: Int = jsonObject["order_id"] as? Int ?? -1
                        let type: Int = jsonObject["type"] as? Int ?? -1
                        let batchId: Int = jsonObject["batch_id"] as? Int ?? 0
                        if msgType == "card_msg", navH5Url.count > 0 {
                            let title = jsonObject["title"] as? String ?? ""
                            var url = ""
                            if YHLoginManager.shared.isLogin() {
                                let token = YHLoginManager.shared.h5Token
                                if navH5Url.contains("?") {
                                    url = navH5Url + "&param=" + token
                                } else {
                                    url = navH5Url + "?param=" + token
                                }
                            } else {
                                url = navH5Url
                            }
                            let vc = YHH5WebViewVC()
                            vc.isFullScreenFlag = false
                            vc.navTitle = title
                            vc.url = url
                            UIViewController.current?.navigationController?.pushViewController(vc)
                            return QYLinkClickActionPolicy.cancel
                        } else if orderID > -1, type > -1 {
                            YHServiceOrderListViewController.jumpToMessageController(type: type, orderId: orderID, batchId)
                            return QYLinkClickActionPolicy.cancel
                        }
                    } else {
                        printLog("ddddd")
                    }
                } else {
                    printLog("JSON字符串不是有效的字典格式")
                }
            } catch {
                printLog("JSON解析错误: \(error)")
            }
        }
        return QYLinkClickActionPolicy.open
    }
    
    private func  updateActionConfig() {
        // 账号登录后是否拉取漫游消息
        customActionConfig?.pullRoamMessage = true
        // 拉取漫游消息条数，默认20条，最大100条
        customActionConfig?.roamMessageLimit = 100
        
        customActionConfig?.linkClickBlock = { linkAddress in
            guard let linkAddress = linkAddress, linkAddress.isValidHttpUrl || linkAddress.isValidHttpsUrl || linkAddress.isValidFileUrl, let url = URL(string: linkAddress) else {
                return self.handelMessageType(linkAddress)
            }
            
            let safariViewController = SFSafariViewController(url: url)
            safariViewController.dismissButtonStyle = .close
            safariViewController.modalPresentationStyle = .fullScreen
            UIViewController.current?.present(safariViewController, animated: true, completion: nil)
            return QYLinkClickActionPolicy.cancel
        }
    }
}

extension YHButlerServiceManager {
    // MARK: - 常用公开方法
    
    /// 灰度打开
    func isUatAllowed() -> Bool {
        let allowed = YHUatHelper.shared.getUatModuleStatusBy(module: .QiYu_Service) == 1
        return allowed
    }
    
    /// 初始化SDK用户信息
    func initSDKAccountAndLastMessage(completion: @escaping (Bool) -> Void) {
        updateSDKAccountInfo(completion: { flag in
            completion(flag)
        })
    }
        
    /// 必须在初始化后调用
    func getLastMessage() -> YHButlerServiceMessage? {
        guard let message = getSessionList().last else {
            return nil
        }
        return YHButlerServiceMessageHandler.handleSessionMessage(sessionMessage: message)
    }
}

extension YHButlerServiceManager {
    // MARK: - 私有方法以及登录登出的处理逻辑
    
    private func isIOS17() -> Bool {
        let systemVersion = UIDevice.current.systemVersion
        let components = systemVersion.split(separator: ".").compactMap { Int($0) }
        guard components.count >= 1 else { return false }
        return components[0] == 17
    }
    
    @objc func sessionGoBack() {
        guard let curentPresentedVC = curentPresentedVC else {
            return
        }
        curentPresentedVC.dismiss(animated: true)
        printLog("$$$$sessionGoBack")
        self.curentPresentedVC = nil
    }
    
    @objc private func didRevUatParam() {
        guard isUatAllowed() else {
            return
        }
        initSDKAccountAndLastMessage { _ in
            // 为了更新状态
            DispatchQueue.main.async {
                NotificationCenter.default.post(name: YhConstant.YhNotification.didQiYuUnReadMsgCountChangeNotification, object: nil)
            }
            
        }
    }
    
    @objc private func loginSuccess() {
        reportHandler.reset()
        guard isUatAllowed() else {
            return
        }
        
        initSDKAccountAndLastMessage { _ in
            // 为了更新状态
            DispatchQueue.main.async {
                NotificationCenter.default.post(name: YhConstant.YhNotification.didQiYuUnReadMsgCountChangeNotification, object: nil)
            }
        }
    }
    
    @objc private func logoutSuccess() {
        // 如果登出时需要关闭聊天页
        /*
        if let controller = QYSDK.shared().sessionViewController(), controller.isViewLoaded && controller.view.window != nil  {
            controller.closeSession(true) { _, _ in
                //
            }
        }
        */
        logoutCurrentUser { _ in
        }
    }
    
    private func updateUserInfo(userId: String, userData: String, completion: @escaping (Bool) -> Void) {
        let userInfo = QYUserInfo()
        userInfo.userId = userId
        userInfo.data = userData
        QYSDK.shared().setUserInfo(userInfo) { [weak self] success, error in
            self?.logUserData(userId: userId, userData: userData, error: error)
            completion(success)
        }
    }

    private func logUserData(userId: String, userData: String, error: Error?) {
        var errorMsg = ""
        if let error = error as? NSError {
            switch error.code {
            case 0:
                errorMsg = "未知错误"
            case 1:
                errorMsg = "错误参数"
            case 2:
                errorMsg = "必须为融合SDK"
            case 3:
                errorMsg = "帐号错误-底层通信IM帐号未登录"
            case 4:
                errorMsg = "userId错误，应与帐号相同"
            case 5:
                errorMsg = "需要登出"
            default:
                break
            }
        }
        let dic: [AnyHashable: Any] = [
            "type": "unicornService",
            "userData": userData,
            "userId": userId,
            "errorMsg": errorMsg
        ]
        YHAliYunLogManager.shared.addModuleName(.logTypeError, remark: dic)
    }
    
    private func setupInfomationForQiYu(completion: @escaping (Bool) -> Void) {
        guard let userId = YHLoginManager.shared.userModel?.id, userId.count > 0 else {
            completion(false)
            return
        }
        getUserInfoViewModel.getUserInfomation(userId) { [weak self] dataString, avatar, groupTmpId, _ in
            // 改变头像
            self?.customUIConfig?.customerHeadImageUrl = avatar
            self?.customUIConfig?.showServiceNickName = true
            self?.groupTmpId = groupTmpId
            self?.updateUserInfo(userId: userId, userData: dataString ?? "", completion: { success in
                completion(success)
            })
        }
    }
    
    /// 启动先判断是否当前登录账号是SDK账号
    private func updateSDKAccountInfo(completion: @escaping (Bool) -> Void) {
        // 如果未登录直接退出设置
        guard let userId = YHLoginManager.shared.userModel?.id, userId.count > 0 else {
            completion(false)
            return
        }
        // 如果新旧账号相同，直接设置用户账号信息
        guard let currentUserID = QYSDK.shared().currentUserID(), currentUserID.count > 0, currentUserID != userId else {
            setupInfomationForQiYu(completion: { done in
                completion(done)
            })
            return
        }
        
        // 如果新旧账号不相同，先登出SDK，再用新账号信息写入，
        logoutCurrentUser { [weak self] _ in
//            if success {
//                self?.setupInfomationForQiYu(completion: { done in
//                    completion(done)
//                })
//            } else {
//                completion(success)
//            }
            
            self?.setupInfomationForQiYu(completion: { done in
                completion(done)
            })
        }
    }
    
    private func logoutCurrentUser(completion: @escaping (Bool) -> Void) {
        QYSDK.shared().logout { flag in
            completion(flag)
        }
    }
}

extension YHButlerServiceManager {
    // MARK: - conversationManager方法
    
    /// 获取所有会话的列表；非平台电商用户，只有一个会话项，平台电商用户，有多个会话项
    func getSessionList() -> [QYSessionInfo] {
        let list = conversationManager?.getSessionList() ?? []
        printLog("$$$$getSessionList \(list.last?.lastMessageText ?? "")  \(list.last?.lastMessageType.rawValue ?? 0)  \(list.last?.lastMessageTimeStamp ?? 0)")
        return list
    }
    
    /// 所有的未读数
    func allUnreadCount() -> Int {
        let count = conversationManager?.allUnreadCount() ?? 0
        printLog("$$$$allUnreadCount \(count)")
        return count
    }
    
    /// 清空未读数
    func clearUnreadCount() {
        conversationManager?.clearUnreadCount()
        // 清空已读消息后刷新最新消息状态
        printLog("$$$$clearUnreadCount")
        // unreadCount = 0
        NotificationCenter.default.post(name: YhConstant.YhNotification.didQiYuUnReadMsgCountChangeNotification, object: nil)
    }
    
}

extension YHButlerServiceManager: QYConversationManagerDelegate {
    // MARK: - QYConversationManagerDelegate
    
    /// 会话列表变化；非平台电商用户，只有一个会话项，平台电商用户，有多个会话项  用这个方法可以拿到最新的完整的消息内容
    func onSessionListChanged(_ sessionList: [QYSessionInfo]!) {
        printLog("$$$$onSessionListChanged \(sessionList.first?.lastMessage?.from ?? "") \(sessionList.first?.lastMessageText ?? "") \(sessionList.first?.lastMessageType.rawValue ?? 0) \(sessionList.first?.lastMessageTimeStamp ?? 0) \(sessionList.first?.lastMessage?.messageId ?? "")")
    }
    
    /// 接收消息（进聊天页面会刷新， 目前不用这个方法打点记录）
    func onReceiveMessage(_ message: QYMessageInfo!) {
        printLog("$$$$onReceiveMessage \(message?.text ?? "") \(message?.type.rawValue ?? 0) \(message?.timeStamp ?? 0)")
        if let latestMessage = getSessionList().last {
            reportHandler.handleMessage(latestMessage)
        }
    }
    
    /// 会话未读数变化（进聊天页面会刷新）
    func onUnreadCountChanged(_ count: Int) {
        printLog("$$$$onUnreadCountChanged \(count)")
        // unreadCount = count
        NotificationCenter.default.post(name: YhConstant.YhNotification.didQiYuUnReadMsgCountChangeNotification, object: nil)
    }
}
