//
//  YHResignDocumentMultiLineTipsCell.swift
//  galaxy
//
//  Created by alexzzw on 2024/10/23.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit
import AttributedString

class YHResignDocumentMultiLineTipsCell: YHResignDocumentCell {
    static let cellReuseIdentifier = "YHResignDocumentMultiLineTipsCell"
    
    private let dotWidth: CGFloat = 3
    
    private var detailLabels: [UILabel] = []
    private var dotImageViews: [UIImageView] = []
    
    private lazy var dotImage: UIImage = {
        let image = UIImage(named: "tip_list_index_dot") ?? UIImage()
        return image
    }()
    
    private lazy var bgView: UIImageView = {
        let view = UIImageView()
        view.image = UIImage(named: "document_prompt_bg_small")
        return view
    }()
    
    private lazy var tipIconView: UIImageView = {
        let view = UIImageView()
        view.image = UIImage(named: "service_step_tips")
        return view
    }()
    
    private lazy var infoTitleLabel: UILabel = {
        let label = UILabel()
        label.font = .PFSC_M(ofSize: 13)
        label.textColor = .brandMainColor
        label.text = "填写指引"
        return label
    }()

    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        setupUI()
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    func setupCellInfo(title: String?, details: [ASAttributedString]) {
        infoTitleLabel.text = title
        detailLabels.forEach {
            $0.removeFromSuperview()
        }
        dotImageViews.forEach {
            $0.removeFromSuperview()
        }
        detailLabels.removeAll()
        dotImageViews.removeAll()
        let count = details.count
        details.enumerated().forEach { [weak self] index, detail in
            guard let self = self else {
                return
            }
            let dotImageView = self.getDotImageView()
            let label = self.getDetailLabel(detail)
            self.dotImageViews.append(dotImageView)
            self.detailLabels.append(label)
            self.subContainerView.addSubview(dotImageView)
            self.subContainerView.addSubview(label)
            label.snp.makeConstraints { make in
                if index == 0 {
                    make.top.equalTo(self.infoTitleLabel.snp.bottom).offset(10)
                } else {
                    make.top.equalTo(self.detailLabels[index - 1].snp.bottom).offset(8)
                }
                make.left.equalTo(dotImageView.snp.right).offset(8)
                make.right.lessThanOrEqualToSuperview().offset(-16)
                if index == count - 1 {
                    make.bottom.equalToSuperview().offset(-16)
                }
            }
            
            dotImageView.snp.makeConstraints { make in
                make.left.equalToSuperview().offset(16)
                make.top.equalTo(label.snp.top).offset(7)
                make.width.height.equalTo(self.dotWidth)
            }
        }
        
    }
}

extension YHResignDocumentMultiLineTipsCell {
    
    private func setupUI() {
        updateCellCorner(.single)
        subContainerView.addSubview(bgView)
        subContainerView.addSubview(tipIconView)
        subContainerView.addSubview(infoTitleLabel)
        subContainerView.sendSubviewToBack(bgView)
        bgView.setContentCompressionResistancePriority(.defaultLow, for: .vertical)
        bgView.setContentCompressionResistancePriority(.defaultLow, for: .horizontal)
        bgView.setContentHuggingPriority(.defaultLow, for: .vertical)
        bgView.setContentHuggingPriority(.defaultLow, for: .horizontal)
        bgView.snp.makeConstraints { make in
            make.edges.equalToSuperview()
        }
        tipIconView.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(16)
            make.top.equalToSuperview().offset(18)
            make.width.height.equalTo(14)
        }
        
        infoTitleLabel.snp.makeConstraints { make in
            make.left.equalTo(tipIconView.snp.right).offset(8)
            make.centerY.equalTo(tipIconView)
            make.bottom.lessThanOrEqualToSuperview().offset(-16)
        }
    }
    
    private func getDetailLabel(_ detail: ASAttributedString) -> UILabel {
        let label = UILabel()
        label.textColor = .mainTextColor70
        label.font = .PFSC_R(ofSize: 12)
        label.numberOfLines = 0
        label.lineBreakMode = .byCharWrapping
        label.attributed.text = detail
        return label
    }
    
    private func getDotImageView() -> UIImageView {
        let imageView = UIImageView()
        imageView.image = dotImage
        return imageView
    }
}
