//
//  YHGalaxyNewsCell.swift
//  galaxy
//
//  Created by Dufet on 2025/4/17.
//  Copyright © 2025 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

class YHGalaxyNewsCell: UITableViewCell {

    static let cellReuseIdentifier = "YHGalaxyNewsCell"
    
    lazy var timeLabel = {
        let label = UILabel()
        label.textColor = UIColor.mainTextColor30
        label.textAlignment = .center
        label.font = UIFont.PFSC_R(ofSize: 12)
        label.text = "00:00"
        return label
    }()
    
    lazy var whiteContentView = {
        let view = UIView()
        view.backgroundColor = .white
        view.layer.cornerRadius = 6.0
        view.clipsToBounds = true
        return view
    }()
    
    lazy var coverImgV: UIImageView = {
        let imgV = UIImageView()
        imgV.contentMode = .scaleAspectFill
        imgV.image = UIImage(named: "msg_news_cover_default")
        imgV.clipsToBounds = true
        return imgV
    }()
    
    lazy var playImgV: UIImageView = {
        let imgV = UIImageView()
        imgV.image = UIImage(named: "msg_news_video_play")
        return imgV
    }()
    
    lazy var titleLabel = {
        let label = UILabel()
        label.textColor = UIColor.mainTextColor
        label.textAlignment = .left
        label.font = UIFont.PFSC_M(ofSize: 15)
        label.text = "这是标题"
        return label
    }()
    
    lazy var descLabel = {
        let label = UILabel()
        label.textColor = UIColor.init(hex: 0x8993A2)
        label.textAlignment = .left
        label.font = UIFont.PFSC_R(ofSize: 12)
        label.numberOfLines = 0
        label.text = "这是描述"
        return label
    }()
    
    lazy var redPointView: UIView = {
        let view = UIView()
        view.backgroundColor = .failColor
        view.layer.cornerRadius = 4.0
        view.clipsToBounds = true
        return view
    }()
    
    required init?(coder: NSCoder) {
        super.init(coder: coder)
    }
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        setupUI()
    }
    
    func updateModel(_ model: YHNewsModel) {
        
        timeLabel.text = formatTimestamp(Double(model.time))
        coverImgV.sd_setImage(with: URL(string: model.cover_img), placeholderImage: UIImage(named: "msg_news_cover_default"))
        titleLabel.text = model.title
        descLabel.text = model.content
        redPointView.isHidden = model.is_read
        playImgV.isHidden = model.news_type != 1
    }
    
    func setupUI() {
        
        self.selectionStyle = .none
        self.backgroundColor = .clear
        contentView.backgroundColor = .clear
        
        contentView.addSubview(timeLabel)
        contentView.addSubview(whiteContentView)
        whiteContentView.addSubview(coverImgV)
        coverImgV.addSubview(playImgV)
        whiteContentView.addSubview(titleLabel)
        whiteContentView.addSubview(descLabel)
        whiteContentView.addSubview(redPointView)
        
        timeLabel.snp.makeConstraints { make in
            make.left.right.equalToSuperview()
            make.top.equalTo(16)
            make.height.equalTo(17)
        }
        
        whiteContentView.snp.makeConstraints { make in
            make.left.equalTo(16)
            make.right.equalTo(-16)
            make.top.equalTo(timeLabel.snp.bottom).offset(12)
            make.bottom.equalTo(0)
        }
        
        coverImgV.snp.makeConstraints { make in
            make.left.right.top.equalToSuperview()
            make.height.equalTo(146)
        }
        
        playImgV.snp.makeConstraints { make in
            make.width.height.equalTo(42)
            make.center.equalToSuperview()
        }
        
        titleLabel.snp.makeConstraints { make in
            make.left.equalTo(16)
            make.top.equalTo(coverImgV.snp.bottom).offset(16)
            make.height.equalTo(21)
        }
        
        redPointView.snp.makeConstraints { make in
            make.left.equalTo(titleLabel.snp.right).offset(6)
            make.right.lessThanOrEqualTo(-6)
            make.width.height.equalTo(8.0)
            make.centerY.equalTo(titleLabel)
        }
        
        descLabel.snp.makeConstraints { make in
            make.left.equalTo(16)
            make.right.bottom.equalTo(-16)
            make.height.lessThanOrEqualTo(34.0)
            make.top.equalTo(titleLabel.snp.bottom).offset(4)
        }
        
    }
    
    func formatTimestamp(_ timestamp: TimeInterval) -> String {
        let currentDate = Date()
        let calendar = Calendar.current
        let currentYear = calendar.component(.year, from: currentDate)
        let currentDay = calendar.component(.day, from: currentDate)
        
        let date = Date(timeIntervalSince1970: timestamp)
        let year = calendar.component(.year, from: date)
        let day = calendar.component(.day, from: date)
        
        let dateFormatter = DateFormatter()
        
        if year == currentYear {
            if day == currentDay {
                dateFormatter.dateFormat = "HH:mm"
            } else {
                dateFormatter.dateFormat = "MM-dd"
            }
        } else {
            dateFormatter.dateFormat = "yyyy-MM-dd"
        }
        
        return dateFormatter.string(from: date)
    }

}
