//
//  YHResourceTableViewCell.swift
//  galaxy
//
//  Created by alexzzw on 2025/9/24.
//  Copyright © 2025 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit
import Kingfisher

class YHResourceTableViewCell: UITableViewCell {
    static let labelFont = UIFont.PFSC_R(ofSize: 14)
    static let labelHeight: CGFloat = 20.0
    static let marginX: CGFloat = 20.0
    static let logoWidth: CGFloat = 60.0
    static let logoToRight: CGFloat = 10.0
    static let marginBetweenVLine: CGFloat = 6.0
    static let widthVLine: CGFloat = 1
    static let arrowWidth: CGFloat = 24
    static let arrowToLeft: CGFloat = 24
    
    static let cellReuseIdentifier = "YHResourceTableViewCell"
    
    // MARK: - 属性
    var resourceModel: YHResourceListModel? {
        didSet {
            updateUI()
        }
    }
        
    // MARK: - UI Components
    
    // 分割线
    lazy var separatorLine: UIView = {
        let view = UIView()
        view.backgroundColor = UIColor.brandGrayColor3
        return view
    }()
    
    lazy var vSeparatorLine: UIView = {
        let view = UIView()
        view.backgroundColor = UIColor.brandGrayColor3
        return view
    }()
    
    // 左侧LOGO
    lazy var logoImageView: UIImageView = {
        let imageView = UIImageView()
        imageView.contentMode = .scaleAspectFill
        imageView.layer.cornerRadius = 4
        imageView.clipsToBounds = true
        imageView.layer.borderColor = UIColor.brandGrayColor3.cgColor
        imageView.layer.borderWidth = 1
        imageView.backgroundColor = UIColor.lightGray.withAlphaComponent(0.3)
        return imageView
    }()
    
    lazy var typeTagIcon: UIImageView = {
        let imageView = UIImageView()
        imageView.contentMode = .scaleAspectFit
        return imageView
    }()
    
    lazy var rightArrow: UIImageView = {
        let imageView = UIImageView(image: UIImage(named: "resource_feed_arrow"))
        imageView.contentMode = .scaleAspectFit
        return imageView
    }()
    
    // 主标题
    lazy var titleLabel: UILabel = {
        let label = UILabel()
        label.font = UIFont.PFSC_M(ofSize: 15)
        label.textColor = UIColor.brandGrayColor8
        label.numberOfLines = 2
        return label
    }()
    
    // 公司名称
    lazy var companyLabel: UILabel = {
        let label = UILabel()
        label.font = UIFont.PFSC_R(ofSize: 14)
        label.textColor = UIColor.gray
        label.numberOfLines = 1
        return label
    }()
    
    // 行业标签
    lazy var industryLabel: UILabel = {
        let label = UILabel()
        label.font = UIFont.PFSC_R(ofSize: 14)
        label.textColor = UIColor.gray
        label.numberOfLines = 1
        return label
    }()

    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        setupUI()
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
}

// MARK: - 私有方法
extension YHResourceTableViewCell {
    
    private func setupUI() {
        backgroundColor = .white
        selectionStyle = .none
        
        contentView.addSubview(logoImageView)
        contentView.addSubview(rightArrow)
        contentView.addSubview(typeTagIcon)
        contentView.addSubview(titleLabel)
        contentView.addSubview(companyLabel)
        contentView.addSubview(industryLabel)
        contentView.addSubview(separatorLine)
        contentView.addSubview(vSeparatorLine)
        
        setupConstraints()
    }
    
    private func setupConstraints() {
        // LOGO约束
        logoImageView.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(YHResourceTableViewCell.marginX)
            make.top.equalToSuperview().offset(24)
            make.width.height.equalTo(YHResourceTableViewCell.logoWidth)
        }
                
        // 右上角标签约束
        typeTagIcon.snp.makeConstraints { make in
            make.left.equalTo(logoImageView.snp.right).offset(YHResourceTableViewCell.logoToRight)
            make.top.equalTo(logoImageView)
            make.height.equalTo(16)
            make.width.equalTo(52)
        }
        
        // 主标题约束
        titleLabel.snp.makeConstraints { make in
            make.left.equalTo(logoImageView.snp.right).offset(YHResourceTableViewCell.logoToRight)
            make.right.equalTo(rightArrow.snp.left)
            make.top.equalTo(typeTagIcon.snp.bottom).offset(4)
        }
        
        rightArrow.snp.makeConstraints { make in
            make.right.equalToSuperview().offset(-YHResourceTableViewCell.marginX)
            make.width.height.equalTo(YHResourceTableViewCell.arrowWidth)
            make.top.equalTo(titleLabel)
        }
        
        // 公司名称约束
        companyLabel.snp.makeConstraints { make in
            make.left.equalTo(titleLabel)
            make.top.equalTo(titleLabel.snp.bottom).offset(4)
        }
        
        vSeparatorLine.snp.makeConstraints { make in
            make.left.equalTo(companyLabel.snp.right).offset(YHResourceTableViewCell.marginBetweenVLine)
            make.centerY.equalTo(companyLabel)
            make.width.equalTo(1)
            make.height.equalTo(8)
        }
        
        // 行业标签约束
        industryLabel.snp.makeConstraints { make in
            make.left.equalTo(vSeparatorLine.snp.right).offset(YHResourceTableViewCell.marginBetweenVLine)
            make.right.lessThanOrEqualTo(rightArrow.snp.left)
            make.top.equalTo(companyLabel)
            make.bottom.equalToSuperview().offset(-32)
        }
        
        // 分割线约束
        separatorLine.snp.makeConstraints { make in
            make.bottom.equalToSuperview()
            make.left.right.equalToSuperview().inset(20)
            make.height.equalTo(0.5)
        }
    }
    
    static func getLabelWidth(_ text: String) -> CGFloat {
        let attrString = NSAttributedString(string: text, attributes: [.font: YHResourceTableViewCell.labelFont])
        let width = attrString.yh_width(containerHeight: YHResourceTableViewCell.labelHeight)
        return width
    }
    
    private func updateUI() {
        guard let model = resourceModel else { return }
        
        // 设置基本信息
        let title = model.title.count > 0 ? model.title : "-"
        let company = model.company_name.count > 0 ? model.company_name : "-"
        let categoryName = model.category_name.count > 0 ? model.category_name : "-"
        titleLabel.text = title
        companyLabel.text = company
        industryLabel.text = categoryName
        var needNext = false
        if let needNewLine = model.needNewLine {
            needNext = needNewLine
        } else {
            let needNewLine = model.calculateNeedNewLine()
            model.needNewLine = needNewLine
            needNext = needNewLine
        }
        if !needNext {
            companyLabel.snp.remakeConstraints { make in
                make.left.equalTo(titleLabel)
                make.top.equalTo(titleLabel.snp.bottom).offset(4)
            }
            
            vSeparatorLine.snp.remakeConstraints { make in
                make.left.equalTo(companyLabel.snp.right).offset(YHResourceTableViewCell.marginBetweenVLine)
                make.centerY.equalTo(companyLabel)
                make.width.equalTo(1)
                make.height.equalTo(8)
            }
            
            // 行业标签约束
            industryLabel.snp.remakeConstraints { make in
                make.left.equalTo(vSeparatorLine.snp.right).offset(YHResourceTableViewCell.marginBetweenVLine)
                make.right.lessThanOrEqualTo(rightArrow.snp.left)
                make.top.equalTo(companyLabel)
                make.bottom.equalToSuperview().offset(-32)
            }
        } else {
            companyLabel.snp.remakeConstraints { make in
                make.left.equalTo(titleLabel)
                make.right.lessThanOrEqualTo(rightArrow.snp.left)
                make.top.equalTo(titleLabel.snp.bottom).offset(4)
            }
            
            vSeparatorLine.snp.remakeConstraints { make in
                make.left.equalTo(companyLabel.snp.left)
                make.centerY.equalTo(industryLabel)
                make.width.equalTo(1)
                make.height.equalTo(8)
            }
                        
            // 行业标签约束
            industryLabel.snp.remakeConstraints { make in
                make.left.equalTo(vSeparatorLine.snp.right).offset(YHResourceTableViewCell.marginBetweenVLine)
                make.right.lessThanOrEqualTo(rightArrow.snp.left)
                make.top.equalTo(companyLabel.snp.bottom).offset(4)
                make.bottom.equalToSuperview().offset(-32)
            }
            
        }
        
        // 设置Logo
        if !model.company_logo.isEmpty {
            logoImageView.kf.setImage(
                with: URL(string: model.company_logo),
                placeholder: UIImage(named: "global_default_image")
            )
        } else {
            logoImageView.image = UIImage(named: "global_default_image")
        }
        
        // 设置类型标签
        if model.type == "service" {
            typeTagIcon.image = UIImage(named: "resource_flag_service")
        } else if model.type == "demand" {
            typeTagIcon.image = UIImage(named: "resource_flag_demand")
        } else {
            typeTagIcon.image = nil
        }
    }
    
}
