//
//  YHResignMaterialCell.swift
//  galaxy
//
//  Created by edy on 2024/9/9.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit
import AttributedString

enum YHMaterialUploadStatus: Int {
    // 全部
    case all = -1
    // 审核中
    case review = 0
    // 已完成
    case finish = 1
    // 已驳回
    case rejected = 2
    // 待上传
    case preUpload = 3
    // 未开始
    case unStarted = 4
}

class YHResignMaterialCell: UITableViewCell {
 
    static let cellReuseIdentifier = "YHResignMaterialCell"

    var model: YHMaterialModel = YHMaterialModel()
    var status: YHCertificateUploadStatus = .preUpload
    var templateBlock: (() -> Void)?
    var whiteView: UIView!
    var nameTextView: UITextView!
    var infoLabel: UILabel!
    var statusLabel: UILabel!
    var rightArrowImgV: UIImageView!
    
    lazy var templateButton: UIButton = {
        let btn = UIButton()
        btn.frame = CGRect(x: 0, y: 0, width: 48, height: 16)
        btn.setTitle("资料模板", for: .normal)
        btn.titleLabel?.font = .PFSC_M(ofSize: 10)
        btn.setTitleColor(.brandMainColor, for: .normal)
        btn.layer.cornerRadius = kCornerRadius3
        btn.clipsToBounds = true
        btn.backgroundColor = .init(hex: 0x2F7EF6, alpha: 0.08)
        btn.addTarget(self, action: #selector(didTemplateBtnClicked), for: .touchUpInside)
        btn.YH_clickEdgeInsets = UIEdgeInsets(top: 20, left: 20, bottom: 20, right: 20)
        return btn
    }()
    
    @objc func didTemplateBtnClicked() {
        if let templateBlock = templateBlock {
            templateBlock()
        }
    }

    required init?(coder: NSCoder) {
        super.init(coder: coder)
    }
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        setupUI()
    }
    
    func updateModel(_ model: YHMaterialModel) {
        
        self.model = model
    
        var titleColor: UIColor = .labelTextColor2
        var title = ""
        
        if model.status == YHMaterialStatus.unOpen.rawValue {
            titleColor = .mainTextColor(alpha: 0.5)
            title = "未开始".local
            
        } else if model.status == YHMaterialStatus.preUpload.rawValue {
            titleColor = .brandMainColor
            title = "待上传".local
            
        } else if model.status == YHMaterialStatus.rejected.rawValue {
            titleColor = .failColor
            title = "已驳回".local
        
        } else if model.status == YHMaterialStatus.review.rawValue {
            titleColor = .init(hex: 0xFF8000)
            title = "审核中".local
        
        } else if model.status == YHMaterialStatus.completed.rawValue {
            titleColor = .init(hex: 0x3CC694)
            title = "已完成".local
        }

        statusLabel.text = title
        statusLabel.textColor = titleColor
        
        let img = UIImage(named: "family_info_status_arrow")?.withRenderingMode(.alwaysTemplate)
        self.rightArrowImgV.image = img
        self.rightArrowImgV.tintColor = titleColor
            
        var result: ASAttributedString = .init(string: "", .font(UIFont.PFSC_M(ofSize: 16)), .foreground(UIColor.mainTextColor))
        let mustTag: ASAttributedString = .init("*", .font(UIFont.PFSC_M(ofSize: 16)), .foreground(UIColor.failColor), .baselineOffset(-2))
        let name = model.getName()
        let cerName: ASAttributedString = .init(string: name, .font(UIFont.PFSC_M(ofSize: 16)), .foreground(UIColor.mainTextColor))
        let template: ASAttributedString = " \(.view(templateButton, .original(.center)))"
        
        // 标题前是否展示红色*
        if model.upload_require == "MUST" {
            result += mustTag
        }
        result += cerName
        
        // 是否展示右边的模板按钮
        let showTemplateBtn = true
        if showTemplateBtn {
            result += template
        }
        nameTextView.attributed.text = result
    }
    
    func setupUI() {
        
        self.selectionStyle = .none
        contentView.backgroundColor = UIColor(hexString: "#F8F8F8")
        
        whiteView = UIView()
        whiteView.layer.cornerRadius = kCornerRadius6
        whiteView.clipsToBounds = true
        whiteView.backgroundColor = .white
        contentView.addSubview(whiteView)
        
        nameTextView = UITextView()
        nameTextView.textColor = UIColor.mainTextColor
        nameTextView.font = UIFont.PFSC_M(ofSize: 16)
        nameTextView.isScrollEnabled = false
        nameTextView.isEditable = false
        nameTextView.isSelectable = false
        whiteView.addSubview(nameTextView)
        
        nameTextView.attributed.text = """
        \("资料名称", .foreground(UIColor.mainTextColor), .font(UIFont.PFSC_M(ofSize: 16))) \(.view(templateButton, .original(.center))))
        """
        
        statusLabel = UILabel()
        statusLabel.textColor = UIColor.mainTextColor(alpha: 0.5)
        statusLabel.textAlignment = .left
        statusLabel.font = UIFont.PFSC_R(ofSize: 14)
        whiteView.addSubview(statusLabel)
        
        rightArrowImgV = UIImageView(image: UIImage(named: "family_info_status_arrow"))
        whiteView.addSubview(rightArrowImgV)
        
        whiteView.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(16)
            make.right.equalToSuperview().offset(-16)
            make.top.equalToSuperview()
            make.bottom.equalToSuperview().offset(-16)
        }
        
        nameTextView.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(16)
            make.top.equalToSuperview().offset(12)
            make.right.equalTo(statusLabel.snp.left).offset(-4)
            make.bottom.equalTo(-12)
        }
        
        statusLabel.snp.makeConstraints { make in
            make.right.equalTo(rightArrowImgV.snp.left).offset(0)
            make.size.equalTo(CGSize(width: 42, height: 20))
            make.centerY.equalToSuperview()
        }
        
        rightArrowImgV.snp.makeConstraints { make in
            make.right.equalToSuperview().offset(-14)
            make.size.equalTo(CGSize(width: 20, height: 20))
            make.centerY.equalToSuperview()
        }
    }
    
    override func hitTest(_ point: CGPoint, with event: UIEvent?) -> UIView? {
        let hitView = super.hitTest(point, with: event)
        if hitView == nameTextView {
            return self
        }
        return hitView
    }
}
