//
//  YHMemberCenterActionCell.swift
//  galaxy
//
//  Created by alexzzw on 2025/2/18.
//  Copyright © 2025 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

class YHMemberCenterActionCell: YHResignDocumentCell {
    static let cellReuseIdentifier = "YHMemberCenterActionCell"
    private let defaultRatio = 180.0 / 311.0
    var actionButtonEvent: (() -> Void)?

    private lazy var leftLabel: UILabel = {
        let label = UILabel()
        label.textColor = UIColor.mainTextColor
        label.font = .PFSC_M(ofSize: 17)
        return label
    }()

    private lazy var rightLabel: UILabel = {
        let label = UILabel()
        label.textColor = UIColor(hexString: "#8993A2")
        label.font = .PFSC_R(ofSize: 12)
        return label
    }()

    private lazy var contentIcon: UIImageView = {
        let view = UIImageView()
        view.image = UIImage(named: "member_privilege_right")
        return view
    }()

    private lazy var actionButton: YHMemberCenterActionButton = {
        let button = YHMemberCenterActionButton(type: .custom)
        button.update(state: .redeemable)
        button.addTarget(self, action: #selector(actionButtonClicked), for: .touchUpInside)
        return button
    }()

    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        setupUI()
    }

    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    @objc private func actionButtonClicked() {
        actionButtonEvent?()
    }

    private func setupUI() {
        subContainerView.radius = 4
        updateCellCorner(.single)
        subContainerView.addSubview(leftLabel)
        subContainerView.addSubview(rightLabel)
        subContainerView.addSubview(contentIcon)
        subContainerView.addSubview(actionButton)

        leftLabel.setContentCompressionResistancePriority(.required, for: .horizontal)
        rightLabel.setContentCompressionResistancePriority(.defaultLow, for: .horizontal)

        leftLabel.snp.makeConstraints { make in
            make.top.equalToSuperview().offset(20)
            make.left.equalToSuperview().offset(16)
        }

        rightLabel.snp.makeConstraints { make in
            make.centerY.equalTo(leftLabel)
            make.left.equalTo(leftLabel.snp.right).offset(2)
            make.right.lessThanOrEqualToSuperview().offset(-8)
        }

        contentIcon.snp.makeConstraints { make in
            make.top.equalTo(rightLabel.snp.bottom).offset(20)
            make.left.equalToSuperview().offset(16)
            make.right.equalToSuperview().offset(-16)
            make.height.equalTo(contentIcon.snp.width).multipliedBy(defaultRatio)
        }

        actionButton.snp.makeConstraints { make in
            make.top.equalTo(contentIcon.snp.bottom).offset(24)
            make.left.equalToSuperview().offset(16)
            make.right.equalToSuperview().offset(-16)
            make.height.equalTo(42)
            make.bottom.equalToSuperview().offset(-16)
        }
    }

    func setupCell(title: String, detail: String, imgStr: String, imageRatio: CGFloat, buttonStr: String, buttonState: YHMemberCenterActionButton.ButtonState) {
        leftLabel.text = title
        rightLabel.text = detail
        contentIcon.image = UIImage(named: imgStr)
        contentIcon.snp.remakeConstraints { make in
            make.top.equalTo(rightLabel.snp.bottom).offset(20)
            make.left.equalToSuperview().offset(16)
            make.right.equalToSuperview().offset(-16)
            make.height.equalTo(contentIcon.snp.width).multipliedBy(imageRatio)
        }
        actionButton.setTitle(buttonStr, for: .normal)
        actionButton.update(state: buttonState)
    }
}

class YHMemberCenterActionButton: UIButton {
    enum ButtonState {
        case redeemable
        case outOfStock
        case memberOnly
    }

    func update(state: ButtonState) {
        switch state {
        case .redeemable:
            backgroundColor = .clear
            setTitleColor(.mainTextColor, for: .normal)
            titleLabel?.font = .PFSC_M(ofSize: 12)
            clipsToBounds = true
            layer.cornerRadius = 2.0
            layer.borderWidth = 1
            layer.borderColor = UIColor(hexString: "#435163")?.cgColor
            isEnabled = true
        case .outOfStock:
            backgroundColor = UIColor(hexString: "#F5F6F8")
            setTitleColor(UIColor(hexString: "#B9C1CC"), for: .normal)
            titleLabel?.font = .PFSC_M(ofSize: 12)
            clipsToBounds = false
            layer.cornerRadius = 0
            layer.borderWidth = 0
            layer.borderColor = nil
            isEnabled = false
        case .memberOnly:
            backgroundColor = UIColor(hexString: "#F5F6F8")
            setTitleColor(UIColor(hexString: "#B9C1CC"), for: .normal)
            titleLabel?.font = .PFSC_M(ofSize: 12)
            clipsToBounds = false
            layer.cornerRadius = 0
            layer.borderWidth = 0
            layer.borderColor = nil
            isEnabled = false
        }
    }
}
