//
//  YHPrincipleUploadStatusCell.swift
//  galaxy
//
//  Created by edy on 2024/8/6.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit
import AttributedString

enum YHPrincipleStatus: Int {
    case preSubmit = 0   // 待提交
    
    case review = 1      // 审核中
    case rejected = 2    // 已驳回
    case pass = 3        // 已通过
    case mail = 4        // 已邮件港府
    
}

class YHPrincipleUploadStatusCell: UITableViewCell {
    
    static let cellReuseIdentifier = "YHPrincipleUploadStatusCell"

    var itemModel:YHPrincipleAgreementModel?
    var items: [String] = []
    var expandClick:((Bool)->())?
    
    var status: YHPrincipleStatus = .review
    
    lazy var whiteView: UIView = {
        let view = UIView()
        view.backgroundColor = .white
        view.layer.cornerRadius = 6.0
        view.clipsToBounds = true
        return view
    }()
    
    lazy var titleLabel : UITextView  = {
        let label = UITextView()
        label.textColor = UIColor(hex: 0x4E4E4E)
        label.font = UIFont.PFSC_R(ofSize: 14)
        label.isScrollEnabled = false
        label.isEditable = false
        label.isSelectable = false
        let mustTag: ASAttributedString = .init("*", .font(UIFont.PFSC_M(ofSize: 17)),.foreground(UIColor.failColor))
        let title: ASAttributedString = .init("《赴港居留同意书》", .font(UIFont.PFSC_M(ofSize: 17)),.foreground(UIColor.mainTextColor))
        let status: ASAttributedString = " \(.view(self.statusLabel, .original(.center)))"
        label.attributed.text = mustTag + title + status
        return label
    }()
    
    lazy var statusLabel: UILabel = {
        let statusLabel = UILabel()
        statusLabel.textAlignment = .center
        statusLabel.font = UIFont.PFSC_M(ofSize:10)
        statusLabel.layer.cornerRadius = kCornerRadius3
        statusLabel.clipsToBounds = true
        statusLabel.text = "审核中"
        statusLabel.backgroundColor = self.getUploadBgColor(status: .review)
        statusLabel.textColor = self.getUploadTitleColor(status: .review)
        statusLabel.frame = CGRectMake(0, 0, 38, 16)
        return statusLabel
    }()
    
    lazy var templateButton: UIButton = {
        let btn = UIButton()
        btn.frame = CGRect(x: 0, y: 0, width: 66, height: 27)
        btn.setTitle("获取模版", for: .normal)
        btn.titleLabel?.font = .PFSC_M(ofSize: 12)
        btn.setTitleColor(.brandMainColor, for: .normal)
        btn.layer.cornerRadius = kCornerRadius3
        btn.clipsToBounds = true
        btn.backgroundColor = UIColor.init(hex: 0xF0F4FB)
        btn.addTarget(self, action: #selector(didTemplateBtnClicked), for: .touchUpInside)
        btn.YH_clickEdgeInsets = UIEdgeInsets(top: 20, left: 20, bottom: 20, right: 20)
        return btn
    }()
    
    lazy var lineView: UIView = {
        let line = UIView()
        line.backgroundColor = .separatorColor
        return line
    }()
    
    lazy var grayLabel: UILabel = {
        let label = UILabel()
        label.numberOfLines = 0
        label.font = UIFont.PFSC_R(ofSize: 12)
        label.textColor = UIColor.labelTextColor2
        return label
    }()
    
    lazy var detailLabel: UILabel = {
        var label = UILabel()
        label.numberOfLines = 0
        return label
    }()
    
    lazy var uploadBtn: UIButton = {
        let btn = UIButton(frame: CGRectMake(0, 0, KScreenWidth-36.0*2.0, 45.0))
        btn.setTitle("上传".local, for: .normal)
        btn.titleLabel?.font = UIFont.PFSC_M(ofSize: 14)
        btn.setTitleColor(UIColor.init(hex: 0x3570DC), for: .normal)
        btn.iconInLeft(spacing: 6)
        btn.setImage(UIImage(named: "form_add_info"), for: .normal)
        btn.addTarget(self, action: #selector(didClickUploadBtn), for: .touchUpInside)
        return btn
    }()
    
    lazy var dotLineLayer:CAShapeLayer = {
        let borderLayer =  CAShapeLayer()
        borderLayer.path = UIBezierPath(roundedRect: CGRect(x: 0, y: 0, width: KScreenWidth-36.0*2, height: 45.0), cornerRadius: 3.0).cgPath
        borderLayer.lineWidth = 1
        //[数组第一个数字表示单个虚线长度，第二个数字表示间隔]
        borderLayer.lineDashPattern = [3,1]  as [NSNumber]?
        //前边是虚线的长度，后边是虚线之间空隙的长度
        borderLayer.lineDashPhase = 0.1
        //实线边框
        borderLayer.fillColor = UIColor.clear.cgColor
        borderLayer.strokeColor = UIColor.brandMainColor.cgColor
        return borderLayer
    }()
    
    lazy var redTipsLabel: UILabel = {
        let label = UILabel()
        label.numberOfLines = 0
        label.font = UIFont.PFSC_R(ofSize: 12)
        label.textColor = UIColor.failColor
        label.text = "请上传《赴港居留同意书》"
        return label
    }()
    
    lazy var tableView: UITableView = {
        let tableView = UITableView(frame: CGRect.zero, style:.grouped)
        tableView.contentInsetAdjustmentBehavior = .never
        tableView.showsVerticalScrollIndicator = false
        tableView.backgroundColor = .clear
        tableView.separatorStyle = .none
        tableView.bounces = false
        tableView.register(YHPrincipleUploadFileCell.self, forCellReuseIdentifier: YHPrincipleUploadFileCell.cellReuseIdentifier)
        tableView.delegate = self
        tableView.dataSource = self
        tableView.isScrollEnabled = false
        return tableView
    }()
    
    @objc func didTemplateBtnClicked() {
        
    }
    
    @objc func didClickUploadBtn() {
        
    }
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        setupUI()
    }
    
    required init?(coder: NSCoder) {
        super.init(coder: coder)
    }
    
    func setupUI() {
        self.contentView.backgroundColor = .clear
        self.backgroundColor = .clear
    }
    
    func updateModel(_ model: YHPrincipleAgreementModel) {
        
        itemModel = model
        items.removeAll()
//        items.append(contentsOf: model.fileList)
        
        contentView.removeSubviews()
        self.dotLineLayer.removeFromSuperlayer()
        
        contentView.addSubview(whiteView)
        whiteView.addSubview(titleLabel)
        whiteView.addSubview(templateButton)
        whiteView.addSubview(lineView)
        whiteView.addSubview(grayLabel)
        whiteView.addSubview(detailLabel)
        whiteView.addSubview(tableView)
        whiteView.addSubview(uploadBtn)
        whiteView.addSubview(redTipsLabel)
        
        grayLabel.text = model.tips
//        detailLabel.attributed.text = model.tips1
        templateButton.isHidden = !model.isNeedTemplate
        uploadBtn.isHidden = !model.isNeedUploadBtn
        if model.isNeedUploadBtn {
            uploadBtn.layer.addSublayer(self.dotLineLayer)
        }

        whiteView.snp.remakeConstraints { make in
            make.top.equalTo(0)
            make.left.equalTo(20)
            make.right.equalTo(-20)
            make.bottom.equalTo(0)
        }
        
        titleLabel.snp.remakeConstraints { make in
            make.top.equalTo(8)
            make.left.equalTo(11)
            make.right.equalTo(model.isNeedTemplate ? -90 : -8)
        }
        
        templateButton.snp.remakeConstraints { make in
            make.top.equalTo(13)
            make.right.equalTo(-16)
            if model.isNeedTemplate {
                make.width.equalTo(66)
            } else {
                make.width.equalTo(0)
            }
            make.height.equalTo(27)
        }
        
        lineView.snp.remakeConstraints { make in
            make.left.equalTo(0)
            make.right.equalTo(0)
            make.top.equalTo(titleLabel.snp.bottom).offset(4)
            make.height.equalTo(1.0)
        }
        
        grayLabel.snp.remakeConstraints { make in
            make.top.equalTo(lineView.snp.bottom).offset(model.tips.isEmpty ? 0 : 16)
            make.left.equalTo(16)
            make.right.equalTo(-16)
            if model.tips.isEmpty {
                make.height.equalTo(0)
            }
        }
        
        detailLabel.snp.remakeConstraints { make in
            make.top.equalTo(grayLabel.snp.bottom).offset(model.tips1.isEmpty ? 0 : 16)
            make.left.equalTo(16)
            make.right.equalTo(-16)
            if model.tips1.isEmpty {
                make.height.equalTo(0)
            }
        }
        
        tableView.snp.remakeConstraints { make in
            make.top.equalTo(detailLabel.snp.bottom).offset(model.fileList.count <= 0 ? 0 : 16)
            make.left.equalTo(0)
            make.right.equalTo(0)
            var listHeight = 0.0
            if model.fileList.count > 0 {
               listHeight = Double(model.fileList.count) * 55.0 + Double(model.fileList.count-1) * 8.0
            }
            make.height.equalTo(listHeight)
        }
        
        uploadBtn.snp.remakeConstraints { make in
            make.left.equalTo(16)
            make.right.equalTo(-16)
            if model.isNeedUploadBtn {
                make.height.equalTo(45)
            } else {
                make.height.equalTo(0)
            }
            make.top.equalTo(tableView.snp.bottom).offset(!model.isNeedUploadBtn  ? 0 : 16)
        }
        
        redTipsLabel.snp.remakeConstraints { make in
            make.left.equalTo(16)
            make.right.equalTo(-16)
            make.top.equalTo(uploadBtn.snp.bottom).offset(model.redTips.isEmpty ? 0 : 16)
            if model.redTips.isEmpty {
                make.height.equalTo(0)
            }
            make.bottom.equalTo(-16)
        }
    }
    
    func getUploadBgColor(status: YHPrincipleStatus) -> UIColor {
        var color = UIColor.black
        if status == .review {
            color = UIColor(hex: 0xFFF5EB)
            
        } else if status == .rejected {
            color = UIColor(hex: 0xFFF2F2)
            
        } else if status == .pass {
            color = UIColor(hex: 0xF0FBF7)
            
        } else if status == .mail {
            color = UIColor(hex: 0xF0F4FB)
        }
        return color
    }
    
    func getUploadTitleColor(status: YHPrincipleStatus) -> UIColor {
        var color = UIColor.black
        if status == .review {
            color = UIColor(hex: 0xFF8000)
            
        } else if status == .rejected {
            color = UIColor(hex: 0xF81D22)
            
        } else if status == .pass {
            color = UIColor(hex: 0x3CC694)
            
        } else if status == .mail {
            color = UIColor.brandMainColor
        }
        return color
    }
}

extension YHPrincipleUploadStatusCell: UITableViewDataSource, UITableViewDelegate {
    
    func numberOfSections(in tableView: UITableView) -> Int {
        return items.count
    }
    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return 1
    }
    
    func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        let cell = tableView.dequeueReusableCell(withIdentifier: YHPrincipleUploadFileCell.cellReuseIdentifier, for: indexPath) as! YHPrincipleUploadFileCell
        return cell
    }
    
    func tableView(_ tableView: UITableView, heightForRowAt indexPath: IndexPath) -> CGFloat {
        return 55.0
    }
    
    func tableView(_ tableView: UITableView, didSelectRowAt indexPath: IndexPath) {
        
    }
    
    func tableView(_ tableView: UITableView, heightForHeaderInSection section: Int) -> CGFloat {
        return 0.001
    }
    
    func tableView(_ tableView: UITableView, viewForHeaderInSection section: Int) -> UIView? {
        return UIView()
    }
    
    func tableView(_ tableView: UITableView, heightForFooterInSection section: Int) -> CGFloat {
        return 8.0
    }
    
    func tableView(_ tableView: UITableView, viewForFooterInSection section: Int) -> UIView? {
        return UIView()
    }
}
