//
//  YHGCSchemeTableViewCell.swift
//  galaxy
//
//  Created by alexzzw on 2024/12/21.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//

import UIKit

class YHGCSchemeTableViewCell: UITableViewCell {
    // MARK: - Properties
    var schemeBlock: (() -> Void)?
    
    var dataSource: YHScemeItemModel? {
        didSet { updateAllViews() }
    }
    
    // MARK: - UI Components
    private lazy var centerView: UIView = {
        let view = UIView()
        view.backgroundColor = .white
        return view
    }()
    
    private lazy var lineImageView: UIImageView = {
        let imageView = UIImageView(image: UIImage(named: "scheme_line_image"))
        return imageView
    }()
    
    private lazy var titleView: UIView = {
        let view = UIView()
        view.backgroundColor = UIColor(hex: 0xd48638)
        return view
    }()
    
    private lazy var titleLabel: UILabel = {
        let label = UILabel()
        label.font = .PFSC_M(ofSize: 17)
        label.textColor = .mainTextColor
        return label
    }()
    
    private lazy var titleButton: UIButton = {
        let button = UIButton(type: .custom)
        button.titleLabel?.font = .PFSC_M(ofSize: 12)
        button.backgroundColor = UIColor(hex: 0xd48638).withAlphaComponent(0.08)
        button.contentHorizontalAlignment = .center
        button.setTitle("资料清单", for: .normal)
        button.setImage(UIImage(named: "scheme_button_icon"), for: .normal)
        button.setTitleColor(UIColor(hex: 0xd48638), for: .normal)
        button.layer.cornerRadius = kCornerRadius3
        button.clipsToBounds = true
        button.addTarget(self, action: #selector(goNext), for: .touchUpInside)
        return button
    }()
    
    private lazy var mainMessageLabel: UILabel = {
        let label = UILabel()
        label.font = .PFSC_R(ofSize: 14)
        label.textColor = .mainTextColor
        label.numberOfLines = 0
        return label
    }()
    
    private lazy var subMessageLabel: UILabel = {
        let label = UILabel()
        label.font = .PFSC_R(ofSize: 14)
        label.textColor = .labelTextColor2
        label.numberOfLines = 0
        return label
    }()
    
    // MARK: - Initialization
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: style, reuseIdentifier: reuseIdentifier)
        setupCell()
    }
    
    required init?(coder: NSCoder) {
        super.init(coder: coder)
        setupCell()
    }
    
    // MARK: - Setup
    private func setupCell() {
        selectionStyle = .none
        setupAppearance()
        setupLayout()
    }
    
    private func setupAppearance() {
        contentView.backgroundColor = UIColor(hex: 0x070E33)
        backgroundColor = UIColor(hex: 0x070E33)
    }
    
    private func setupLayout() {
        // Center View
        contentView.addSubview(centerView)
        // Line Image View
        contentView.addSubview(lineImageView)
        // Title View
        contentView.addSubview(titleView)
        // Title Label
        centerView.addSubview(titleLabel)
        // Title Button
        centerView.addSubview(titleButton)
        // Main Message Label
        centerView.addSubview(mainMessageLabel)
        // Sub Message Label
        centerView.addSubview(subMessageLabel)
        centerView.snp.makeConstraints { make in
            make.left.equalTo(8)
            make.right.equalTo(-8)
            make.top.bottom.equalToSuperview()
        }
        
        lineImageView.snp.makeConstraints { make in
            make.left.equalTo(28)
            make.top.equalTo(0)
            make.height.equalTo(1)
            make.right.equalTo(-28)
        }
        
        titleView.snp.makeConstraints { make in
            make.left.equalTo(28)
            make.top.equalTo(29)
            make.height.equalTo(16.5)
            make.width.equalTo(3.5)
        }
        
        titleLabel.snp.makeConstraints { make in
            make.centerY.equalTo(titleView)
            make.left.equalTo(titleView.snp.right).offset(4)
            make.height.equalTo(24)
            make.width.equalTo(150)
        }
       
        titleButton.snp.makeConstraints { make in
            make.centerY.equalTo(titleView)
            make.right.equalTo(-28)
            make.height.equalTo(27)
            make.width.equalTo(76)
        }
        
        mainMessageLabel.snp.makeConstraints { make in
            make.top.equalTo(69)
            make.left.equalTo(28)
            make.right.equalTo(-28)
        }
        
        subMessageLabel.snp.makeConstraints { make in
            make.top.equalTo(mainMessageLabel.snp.bottom).offset(12)
            make.left.equalTo(28)
            make.right.equalTo(-28)
            make.bottom.equalToSuperview().offset(-26)
        }
    }
    
    // MARK: - Update UI
    private func updateAllViews() {
        guard let model = dataSource else { return }
        
        titleButton.isHidden = model.buttonTitle.isEmpty
        titleLabel.text = model.title
        titleButton.setTitle(model.buttonTitle, for: .normal)
        titleButton.iconInRight(with: 2)
        lineImageView.isHidden = model.hideTopLine
        
        updateMainMessage(model)
        updateSubMessage(model)
    }
    
    private func updateMainMessage(_ model: YHScemeItemModel) {
        let paragraphStyle = NSMutableParagraphStyle()
        paragraphStyle.lineSpacing = 4
        let newString = model.mainMessage.highlighted(
            keyWords: model.lightMessage,
            highlightColor: UIColor(hex: 0xd48638),
            highlightFont: model.lightFont,
            paragraph: paragraphStyle,
            normalFont: UIFont.PFSC_R(ofSize: 14)
        )
        mainMessageLabel.attributedText = newString
    }
    
    private func updateSubMessage(_ model: YHScemeItemModel) {
        let paragraphStyle = NSMutableParagraphStyle()
        paragraphStyle.lineSpacing = 4
        
        let attributes: [NSAttributedString.Key: Any] = [
            .paragraphStyle: paragraphStyle,
            .font: UIFont.PFSC_R(ofSize: 14)
        ]
        
        subMessageLabel.attributedText = NSAttributedString(
            string: model.subMessage,
            attributes: attributes
        )
    }
    
    @objc private func goNext() {
        schemeBlock?()
    }
}
