//
//  YHCertificateActionSheetView.swift
//  galaxy
//
//  Created by edy on 2024/3/12.
//  Copyright © 2024 https://www.galaxy-immi.com. All rights reserved.
//
/*
 【Usage】
 let items = [YHCertificateEditItem(type:.rename, title:"重命名"),
              YHCertificateEditItem(type:.preview, title:"预览"),
              YHCertificateEditItem(type:.delete, title:"删除"),
              YHCertificateEditItem(type:.cancel, title:"取消")]
 YHCertificateEditSheetView.sheetView(items:items) {
     [weak self] editType in
     guard let self = self else { return }
     print("editType:\(editType.rawValue)")
 }.show()
 
 */

import UIKit

enum YHCertificateEditType:Int {
    case rename = 1
    case preview = 2
    case delete = 3
    case cancel = 4
    case reupload = 5
    case sendEmail = 6
}

class YHCertificateEditItem {
    var type: YHCertificateEditType
    var title: String
    
    init(type:YHCertificateEditType, title: String,itemTitle : String? = nil) {
        self.type = type
        self.title = title
    }
}

class YHCertificateEditItemView: UIView {
    
    let item: YHCertificateEditItem
    lazy var topLine:UIView = {
        let view = UIView()
        view.backgroundColor = .separatorColor
        return view
    }()
    
    lazy var titleLabel: UILabel = {
        let label = UILabel()
        label.textColor = UIColor.mainTextColor
        label.textAlignment = .center
        label.font = UIFont.PFSC_M(ofSize:14)
        return label
    }()
    
    init(item: YHCertificateEditItem) {
        self.item = item
        super.init(frame:CGRect(x: 0, y: 0, width: KScreenWidth, height: 52.0))
        self.tag = item.type.rawValue
        self.createUI()
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    func createUI() {
        
        self.addSubview(titleLabel)
        self.addSubview(topLine)
        self.titleLabel.text = item.title
        
        if self.item.type == .delete {
            titleLabel.textColor = .failColor
            
        } else if self.item.type == .cancel {
            titleLabel.textColor = UIColor(hexString:"#909090")
            
        } else {
            titleLabel.textColor = .mainTextColor
        }
        
        titleLabel.snp.makeConstraints { make in
            make.height.equalTo(20)
            make.left.right.equalToSuperview()
            make.centerY.equalToSuperview()
        }
        
        topLine.snp.makeConstraints { make in
            make.left.equalToSuperview().offset(24)
            make.right.equalToSuperview().offset(-24)
            make.top.equalToSuperview()
            make.height.equalTo(0.5)
        }
    }
}

class YHCertificateEditSheetView: UIView {
    
    var items:[YHCertificateEditItem]
    var action:((YHCertificateEditType)->())?
    private var sheetTitle : String?

    lazy var blackMaskView: UIView = {
        let view = UIView(frame:UIScreen.main.bounds)
        view.backgroundColor = UIColor(hex: 0x0F1214, alpha: 0.5)
        let tap = UITapGestureRecognizer(target: self, action: #selector(dismiss))
        view.addGestureRecognizer(tap)
        return view
    }()
    
    lazy var whiteContentView: UIView = {
        let view = UIView()
        view.backgroundColor = .white
        return view
    }()
    
    lazy var titleLabel: UILabel = {
        let label = UILabel()
        label.textColor = UIColor.mainTextColor
        label.textAlignment = .center
        label.font = UIFont.PFSC_B(ofSize:17)
        label.text = "请选择操作"
        return label
    }()
    
    lazy var closeBtn: UIButton = {
        let btn = UIButton()
        btn.setImage(UIImage(named:"my_cer_sheet_close"), for: .normal)
        btn.addTarget(self, action: #selector(dismiss), for: .touchUpInside)
        return btn
    }()
    
    
    init(items:[YHCertificateEditItem],sheetTitle : String? = nil) {
        self.items = items
        self.sheetTitle = sheetTitle
        super.init(frame:UIScreen.main.bounds)
        createUI()
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    static func sheetView(items:[YHCertificateEditItem],sheetTitle : String? = nil, action:((YHCertificateEditType)->())?) -> YHCertificateEditSheetView {
        let view = YHCertificateEditSheetView(items:items,sheetTitle: sheetTitle)
        view.action = action
        return view
    }
    
    func createUI() {
        
        self.addSubview(blackMaskView)
        self.addSubview(whiteContentView)
        
        let corner = UIRectCorner(rawValue: UIRectCorner.topLeft.rawValue | UIRectCorner.topRight.rawValue)
        let path = UIBezierPath(roundedRect:bounds,byRoundingCorners: corner,cornerRadii:CGSizeMake(kCornerRadius6, kCornerRadius6));
        let layer = CAShapeLayer();
        layer.frame = whiteContentView.bounds;
        layer.path = path.cgPath;
        whiteContentView.layer.mask = layer;
        
        whiteContentView.addSubview(titleLabel)
        whiteContentView.addSubview(closeBtn)
        
        blackMaskView.snp.makeConstraints { make in
            make.edges.equalToSuperview()
        }
        
        whiteContentView.snp.makeConstraints { make in
            make.left.right.bottom.equalToSuperview()
            make.height.equalTo(52.0 + CGFloat(items.count) * 52.0 + k_Height_safeAreaInsetsBottom())
        }
        
        titleLabel.snp.makeConstraints { make in
            make.height.equalTo(52)
            make.left.right.top.equalToSuperview()
        }
        
        if let title = sheetTitle {
            titleLabel.text = title
        }
        
        closeBtn.snp.makeConstraints { make in
            make.centerY.equalTo(titleLabel)
            make.right.equalToSuperview().offset(-20)
            make.width.height.equalTo(24)
        }
        
        for (index, item) in items.enumerated() {
            let itemView = YHCertificateEditItemView(item:item)
            whiteContentView.addSubview(itemView)
            let tap = UITapGestureRecognizer(target:self, action:#selector(action(_:)))
            itemView.addGestureRecognizer(tap)
            itemView.snp.makeConstraints { make in
                make.left.right.equalToSuperview()
                make.height.equalTo(52.0)
                make.top.equalTo(titleLabel.snp.bottom).offset(index * 52)
            }
        }
    }
}

extension YHCertificateEditSheetView {
    
    func show() {
        UIApplication.shared.yhKeyWindow()?.addSubview(self)
    }
    
    @objc func dismiss() {
        self.removeFromSuperview()
    }
    
    @objc func action(_ sender:UITapGestureRecognizer) {
        
        if let targetView = sender.view {
            let tag = targetView.tag
            if let type = YHCertificateEditType(rawValue:tag), let action = action {
                action(type)
                dismiss()
            }
        }
    }
}
